/*
 * Part of the PiyoPiyoJ project.
 * (PiyoPiyo Java)
 *
 *        DO WHATEVER YOU WANT TO PUBLIC LICENSE 
 *                    Version 2, December 2004 
 *
 * Modified by 20kdc <asdd2808@gmail.com> to remove expletives
 * Original copyright (C) 2004 Sam Hocevar <sam@hocevar.net> 
 *
 * Everyone is permitted to copy and distribute verbatim or modified 
 * copies of this license document, and changing it is allowed as long 
 * as the name is changed. 
 *
 *            DO WHATEVER YOU WANT TO PUBLIC LICENSE 
 *   TERMS AND CONDITIONS FOR COPYING, DISTRIBUTION AND MODIFICATION 
 *
 *  0. You just DO WHATEVER YOU WANT TO.
 */
package piyopiyoj.games.ui;

import java.util.function.Consumer;

import gabien.*;
import gabien.wsi.*;
import gabien.render.*;
import gabien.text.IImmFixedSizeFont;
import gabien.text.RenderedTextChunk;
import gabien.text.TextTools;
import gabien.ui.*;
import gabien.uslx.append.*;
import libpiyo.PiyoPiyoFile;
import libpiyo.PiyoPiyoFrame;

public class UINotesDisplay extends UIElement implements OldMouseEmulator.IOldMouseReceiver {
    public final PiyoPiyoFile file;
    public final Consumer<Boolean> scrollWheel;
    public static final int fingerLen = 24;
    public static final int tileSize = 12;
    public int xOffset, track;
    public int selectedStart = 1, selectedEnd = 0;
    public OldMouseEmulator emulation = new OldMouseEmulator(this);

    public NoteClickCallback toolCallback = null;
    public boolean[] fingerInvert = new boolean[24];
    public RenderedTextChunk[] fingerTextA = new RenderedTextChunk[24];
    public RenderedTextChunk[] fingerTextB = new RenderedTextChunk[24];
    public RenderedTextChunk[] fingerTextAI = new RenderedTextChunk[24];
    public RenderedTextChunk[] fingerTextBI = new RenderedTextChunk[24];

    public interface NoteClickCallback {
        // Note that frameIndex doesn't include the track offset.
        void onClick(int frameIndex, int pos, int button);

        void onDrag(int frameIndex, int pos);

        void onRelease();

        void onFingerClick(int i, int button);

        void prePC();
        void postPC();
    }

    public UINotesDisplay(PiyoPiyoFile f, Consumer<Boolean> sw) {
        super(640, 12 * (8 + 24));
        file = f;
        scrollWheel = sw;
        // Thanks to Jazz Jackalope for the idea and letter list.
        String[] keyA = {"B", "A#", "A", "G#", "G", "F#", "F",
                "E", "D#", "D", "C#", "C", "B", "A#", "A", "G#",
                "G", "F#", "F", "E", "D#", "D", "C#", "C"};
        // Since zxin wanted it...
        String[] keyB = new String[] {
                "", "", "", "", "", "", "",
                "", "", "", "Cs", "Cs!", "Cl", "Cl!", "HH", "HH!",
                "", "", "X3", "X3!", "X2", "X2!", "X1", "X1!"
        };
        IImmFixedSizeFont f8 = GaBIEn.engineFonts.f8;
        for (int i = 0; i < keyA.length; i++) {
            fingerTextA[i] = TextTools.renderString(keyA[i], f8, false);
            fingerTextB[i] = TextTools.renderString(keyB[i], f8, false);
            fingerTextAI[i] = TextTools.renderString(keyA[i], f8, true);
            fingerTextBI[i] = TextTools.renderString(keyB[i], f8, true);
        }
    }

    @Override
    public void update(double deltaTime, boolean selected, IPeripherals peripherals) {
        Size elementBounds = getSize();

        // Autoexpand the file based upon what the user can see.
        int requiredSize = xOffset + (elementBounds.width / tileSize);
        if (file.getFrameCount() < requiredSize)
            file.resize(requiredSize + 64); // extra 64 for good measure, no point reallocating all the time
    }

    @Override
    public void renderLayer(IGrDriver igd, UILayer layer) {
        if (layer != UILayer.Content)
            return;
        igd.clearAll(0, 0, 0);
        IImage ui = GaBIEn.getImageCKEx("ui.png", false, true, 0, 0, 0);
        Size elementBounds = getSize();

        // Draw the notes on the left side.

        boolean[] sharp = {false, true, false, true, false, true, false,
                false, true, false, true, false, false, true, false, true,
                false, true, false, false, true, false, true, false};

        for (int py = 0; py < (elementBounds.height / tileSize); py++)
            if (py < fingerInvert.length) {
                igd.blitImage(sharp[py] ? fingerLen : 0, tileSize, fingerLen, tileSize, 0, py * tileSize, ui);
                RenderedTextChunk src;
                if (track != 3) {
                    if (fingerInvert[fingerInvert.length - (py + 1)]) {
                        src = fingerTextAI[py];
                    } else {
                        src = fingerTextA[py];
                    }
                } else {
                    if (fingerInvert[fingerInvert.length - (py + 1)]) {
                        src = fingerTextBI[py];
                    } else {
                        src = fingerTextB[py];
                    }
                }
                src.renderRoot(igd, 1, 2 + (py * tileSize) + src.highestAscent);
            }
        // Draw the note grid itself.
        boolean firstColumn = true;
        for (int px = 0; px < ((elementBounds.width - fingerLen) / tileSize); px++) {
            if (px + xOffset < 0)
                continue;
            PiyoPiyoFrame frameA = file.getFrameOrNull(px + xOffset, 0);
            PiyoPiyoFrame frameB = file.getFrameOrNull(px + xOffset, 1);
            PiyoPiyoFrame frameC = file.getFrameOrNull(px + xOffset, 2);
            PiyoPiyoFrame frameD = file.getFrameOrNull(px + xOffset, 3);
            PiyoPiyoFrame frameCurrent = file.getFrameOrNull(px + xOffset, track);
            if (frameA == null || frameB == null || frameC == null || frameD == null || frameCurrent == null)
                continue;
            IImmFixedSizeFont f8 = GaBIEn.engineFonts.f8;
            for (int py = 0; py < (elementBounds.height / tileSize); py++) {
                if (py == 24) {
                    boolean aBar = ((px + xOffset) & 15) == 0;
                    boolean flash = (((px + xOffset) >> 2) & 1) == 0;
                    if (!flash)
                        igd.blitImage(0, 0, tileSize, tileSize, fingerLen + (px * tileSize), py * tileSize, ui);
                    if (!aBar)
                        continue;
                    String tx = Integer.toString((px + xOffset) >> 4);
                    int markerX = fingerLen + (px * tileSize);
                    int markerY = py * tileSize;
                    markerY += f8.getLineHeight();
                    f8.drawLine(igd, markerX, markerY, tx, 255, 255, 255, 255);
                    continue;
                }
                if (py >= 25) {
                    if (py >= 32)
                        continue;
                    int idx = 5;
                    int idxY = 0;
                    if (firstColumn) {
                        if (py == 25) {
                            idx = 6;
                            idxY = 1;
                        }
                        if (py == 31) {
                            idx = 5;
                            idxY = 1;
                        }
                    }
                    igd.blitImage(tileSize * idx, tileSize * idxY, tileSize, tileSize, fingerLen + (px * tileSize), py * tileSize, ui);
                    if (frameCurrent.panValue == (py - 24))
                        igd.blitImage(tileSize, 0, tileSize, tileSize, fingerLen + (px * tileSize), py * tileSize, ui);
                    continue;
                }
                boolean flash = sharp[py];
                igd.blitImage(flash ? (tileSize * 6) : 0, 0, tileSize, tileSize, fingerLen + (px * tileSize), py * tileSize, ui);
                if (frameCurrent.hitNotes[23 - py]) {
                    igd.blitImage(tileSize, 0, tileSize, tileSize, fingerLen + (px * tileSize), py * tileSize, ui);
                } else {
                    if (frameA.hitNotes[23 - py] || frameB.hitNotes[23 - py] || frameC.hitNotes[23 - py] || frameD.hitNotes[23 - py]) {
                        igd.blitImage(tileSize * 4, tileSize, tileSize, tileSize, fingerLen + (px * tileSize), py * tileSize, ui);
                    }
                }
                if (file.loopEnd == (px + xOffset))
                    igd.blitImage(tileSize * 2, 0, tileSize, tileSize, fingerLen + (px * tileSize), py * tileSize, ui);
                if (file.loopStart == (px + xOffset))
                    igd.blitImage(tileSize * 3, 0, tileSize, tileSize, fingerLen + (px * tileSize), py * tileSize, ui);
                if ((px + xOffset) >= selectedStart)
                    if ((px + xOffset) <= selectedEnd)
                        igd.blitImage(tileSize * 4, 0, tileSize, tileSize, fingerLen + (px * tileSize), py * tileSize, ui);
            }
            firstColumn = false;
        }
    }

    @Override
    public void handleClick(int x, int y, int button) {
        if (x < fingerLen) {
            if ((y / tileSize) >= 24)
                return;
            toolCallback.onFingerClick(23 - (y / tileSize), button);
            return;
        }
        int ox = ((x - fingerLen) / tileSize) + xOffset;
        if (ox < 0)
            return;

        int yds = y / tileSize;
        if (yds == 24)
            return;
        if (yds >= 25) {
            if (yds < 32)
                adjustPan(ox, yds);
            return;
        }
        if (toolCallback != null)
            toolCallback.onClick(ox, 23 - yds, button);
    }

    @Override
    public void handleDrag(int x, int y) {
        if (x < fingerLen)
            return;
        int ox = ((x - fingerLen) / tileSize) + xOffset;
        if (ox < 0)
            return;

        int yds = y / tileSize;
        if (yds >= 24)
            return;
        if (toolCallback != null)
            toolCallback.onDrag(ox, 23 - yds);
    }

    @Override
    public void handleRelease(int x, int y) {
        if (toolCallback != null)
            toolCallback.onRelease();
    }

    @Override
    public void handleMousewheel(int x, int y, boolean north) {
        scrollWheel.accept(north);
    }

    @Override
    public IPointerReceiver handleNewPointer(IPointer state) {
        return emulation;
    }

    private void adjustPan(int ox, int i) {
        if (toolCallback != null)
            toolCallback.prePC();
        PiyoPiyoFrame ppf = file.getFrameOrNull(ox, track);
        if (ppf.panValue == i - 24) {
            ppf.panValue = 0;
        } else {
            ppf.panValue = i - 24;
        }
        if (toolCallback != null)
            toolCallback.postPC();
    }

}
