/*
 * Part of the PiyoPiyoJ project.
 * (PiyoPiyo Java)
 *
 *        DO WHATEVER YOU WANT TO PUBLIC LICENSE 
 *                    Version 2, December 2004 
 *
 * Modified by 20kdc <asdd2808@gmail.com> to remove expletives
 * Original copyright (C) 2004 Sam Hocevar <sam@hocevar.net> 
 *
 * Everyone is permitted to copy and distribute verbatim or modified 
 * copies of this license document, and changing it is allowed as long 
 * as the name is changed. 
 *
 *            DO WHATEVER YOU WANT TO PUBLIC LICENSE 
 *   TERMS AND CONDITIONS FOR COPYING, DISTRIBUTION AND MODIFICATION 
 *
 *  0. You just DO WHATEVER YOU WANT TO.
 */

package piyopiyoj.games;

import gabien.*;
import gabien.ui.*;
import gabien.ui.elements.UILabel;
import gabien.ui.elements.UINumberBox;
import gabien.ui.elements.UIPublicPanel;
import gabien.ui.elements.UIScrollbar;
import gabien.ui.elements.UITextButton;
import gabien.ui.layouts.UISplitterLayout;
import gabien.uslx.append.*;
import gabien.wsi.*;
import libpiyo.LibPiyo;
import libpiyo.PiyoPiyoFrame;
import piyopiyoj.Game;
import libpiyo.export.*;
import piyopiyoj.games.edittools.*;
import piyopiyoj.games.ui.UINotesDisplay;
import piyopiyoj.games.ui.UIPlayerControls;
import piyopiyoj.games.ui.UIWaveTrackHeaderEditor;

import java.io.InputStream;
import java.io.OutputStream;
import java.util.*;
import java.util.function.Consumer;

public class NotesheetGame extends UIElement.UIPanel {

    public final Game root;

    /*
     * Schematic, 24TH MAY 2022:
     *
     * 0: this
     *  1: internal UISplitterLayout attached to bottom 'major'
     *   2: etcetc 'ts'
     *    3: probably something involving a UIScrollLayout (???) 'trackAndPlayC'
     *   4: internal UISplitterLayout attached to bottom 'minor'
     *
     * +-------------------+
     * | trackHeaderEditor |
     * |  OR               |
     * | noteDisplay       |
     * +0-----------------0+
     * | trackScroll       |
     * +2--------+--------2+
     * | topPanC 3 playerC |
     * +1--------+--------1+
     * | toolVS            |
     * +4--------+--------4+
     * | fileControls      | <- complicated
     * +-------------------+
     */

    public UINotesDisplay noteDisplay;
    private UIElement[] trackHeaderEditor = new UIElement[4];

    private EditTool selectedTool;

    private double keybindTickTime = 0;

    private UIScrollbar trackScroll = new UIScrollbar(false, 8);
    private double lastScrollPoint = 0;

    // -- topPanC

    private UITextButton track1 = new UITextButton("1", 16, new Runnable() {
        @Override
        public void run() {
            if (editTrack.state)
                attachmentTrackHead(false);
            noteDisplay.track = 0;
            if (editTrack.state)
                attachmentTrackHead(true);
        }
    }).togglable(true);
    private UITextButton track2 = new UITextButton("2", 16, new Runnable() {
        @Override
        public void run() {
            if (editTrack.state)
                attachmentTrackHead(false);
            noteDisplay.track = 1;
            if (editTrack.state)
                attachmentTrackHead(true);
        }
    }).togglable(false);
    private UITextButton track3 = new UITextButton("3", 16, new Runnable() {
        @Override
        public void run() {
            if (editTrack.state)
                attachmentTrackHead(false);
            noteDisplay.track = 2;
            if (editTrack.state)
                attachmentTrackHead(true);
        }
    }).togglable(false);
    private UITextButton trackP = new UITextButton("P", 16, new Runnable() {
        @Override
        public void run() {
            if (editTrack.state)
                attachmentTrackHead(false);
            noteDisplay.track = 3;
            if (editTrack.state)
                attachmentTrackHead(true);
        }
    }).togglable(false);

    private UITextButton editTrack = new UITextButton("Edit", 16, new Runnable() {
        @Override
        public void run() {
            if (!editTrack.state) {
                attachmentTrackHead(false);
            } else {
                attachmentTrackHead(true);
            }
        }
    }).togglable(false);

    private UISplitterLayout topPanCB = new UISplitterLayout(track1, track2, false, 0);
    private UISplitterLayout topPanCC = new UISplitterLayout(track3, trackP, false, 0);
    private UISplitterLayout topPanCA = new UISplitterLayout(topPanCB, topPanCC, false, 0);
    private UISplitterLayout topPanC = new UISplitterLayout(topPanCA, editTrack, false, 0);

    // -- fileControls

    private UITextButton loadFile = new UITextButton("Load", 16, new Runnable() {
        @Override
        public void run() {
            root.fileDialog(false, new Consumer<String>() {
                @Override
                public void accept(String s) {
                    if (s == null)
                        return;
                    root.playerLock.lock();
                    root.playerControls.setPlaying(false);
                    root.playerControls.setCurrentFrame(0);
                    root.playerControls.resetSynth();
                    root.playerLock.unlock();
                    root.undoBuffer.clear();
                    root.redoBuffer.clear();
                    root.currentModification = null;
                    try {
                        InputStream fis = GaBIEn.getInFile(s);
                        root.rememberFileName = s;
                        root.coreFile.loadFile(fis);
                        fis.close();
                        // Stuff needs to be reloaded.
                        MidOut.setup(s);
                        root.switcher(new NotesheetGame(root));
                    } catch (Exception e) {
                        root.coreFile.reset();
                        root.switchError(e, new NotesheetGame(root));
                        e.printStackTrace();
                    }
                    // either way, we just switched files, so reset synth
                    root.playerControls.resetSynth();
                }
            }, "Load");
        }
    });
    private UITextButton saveFile = new UITextButton("Save", 16, new Runnable() {
        @Override
        public void run() {
            root.playerLock.lock();
            root.playerControls.resetSynth();
            root.coreFile.trimEnd();
            root.playerControls.setCurrentFrame(root.playerControls.getCurrentFrame()); // Just in case
            root.playerLock.unlock();
            root.fileDialog(true, new Consumer<String>() {
                @Override
                public void accept(String s) {
                    if (s == null)
                        return;
                    try {
                        OutputStream fos = GaBIEn.getOutFile(s);
                        root.rememberFileName = s;
                        root.coreFile.saveFile(fos);
                        fos.close();
                    } catch (Exception e) {
                        e.printStackTrace();
                        root.switchError(e, NotesheetGame.this);
                    }
                }
            }, "Save");
        }
    });
    private UITextButton configP = new UITextButton("Scale+", 16, new Runnable() {
        @Override
        public void run() {
            root.switcher(new ConfirmGame(root, NotesheetGame.this, "Are you sure?\nThis could make the application unusable, causing you to lose unsaved work.\nIn the event something does go wrong, check the readme!", new Runnable() {
                @Override
                public void run() {
                    root.toggleScale(true);
                }
            }));
        }
    });
    private UITextButton configX = new UITextButton("-", 16, new Runnable() {
        @Override
        public void run() {
            root.toggleScale(false);
        }
    });
    private UITextButton configFS = new UITextButton("FS", 16, new Runnable() {
        @Override
        public void run() {
            root.toggleFullscreen();
        }
    });

    private UINumberBox timeFac = new UINumberBox(2, 16);
    private UITextButton increaseTime = new UITextButton("Tx", 16, new Runnable() {
        @Override
        public void run() {
            timeFac.setNumber(Math.max(1, timeFac.getNumber()));
            root.startModifications("Time x " + timeFac.getNumber());
            int fac = Math.max(1, (int) timeFac.getNumber());
            PiyoPiyoFrame[] oldFrames = root.coreFile.getAllFrames();
            PiyoPiyoFrame[] newTime = new PiyoPiyoFrame[oldFrames.length * fac];
            for (int i = 0; i < newTime.length; i += fac) {
                newTime[i] = new PiyoPiyoFrame(oldFrames[i / fac]);
                for (int j = 1; j < fac; j++)
                    newTime[i + j] = new PiyoPiyoFrame();
            }
            root.playerLock.lock();
            root.coreFile.setAllFrames(newTime);
            root.coreFile.loopStart *= fac;
            root.coreFile.loopEnd *= fac;
            root.coreFile.musicWait /= fac;
            root.playerControls.setCurrentFrame(root.playerControls.getCurrentFrame() * fac);
            root.playerLock.unlock();
            root.endModifications();
            refresh();
        }
    });
    private UITextButton decreaseTime = new UITextButton("T/", 16, new Runnable() {
        @Override
        public void run() {
            timeFac.setNumber(Math.max(1, timeFac.getNumber()));
            root.startModifications("Time / " + timeFac.getNumber());
            int fac = Math.max(1, (int) timeFac.getNumber());
            // The maths on this gets more difficult.
            PiyoPiyoFrame[] oldFrames = root.coreFile.getAllFrames();
            int totalInputCols = oldFrames.length / 4;
            int totalOutputCols = totalInputCols / fac;
            PiyoPiyoFrame[] newTime = new PiyoPiyoFrame[totalOutputCols * 4];
            for (int i = 0; i < totalOutputCols; i++) {
                for (int j = 0; j < 4; j++) {
                    PiyoPiyoFrame frame = new PiyoPiyoFrame();
                    newTime[i + (j * totalOutputCols)] = frame;
                    for (int k = 0; k < fac; k++)
                        frame.merge(oldFrames[(i * fac) + k + (j * totalInputCols)]);
                }
            }
            root.playerLock.lock();
            root.coreFile.setAllFrames(newTime);
            root.coreFile.loopStart /= fac;
            root.coreFile.loopEnd /= fac;
            root.coreFile.musicWait *= fac;
            root.playerControls.setCurrentFrame(root.playerControls.getCurrentFrame() / fac);
            root.playerLock.unlock();
            root.endModifications();
            refresh();
        }
    });
    private UITextButton undo = new UITextButton("Undo", 16, new Runnable() {
        @Override
        public void run() {
            root.undoModifications();
            refresh();
        }
    });
    private UITextButton redo = new UITextButton("Redo", 16, new Runnable() {
        @Override
        public void run() {
            root.redoModifications();
            refresh();
        }
    });

    private UITextButton clearDoBuffer = new UITextButton("X", 16, new Runnable() {
        @Override
        public void run() {
            root.switcher(new ConfirmGame(root, NotesheetGame.this, "Are you sure you want to clear the undo/redo buffers?", new Runnable() {
                @Override
                public void run() {
                    root.undoBuffer.clear();
                    root.redoBuffer.clear();
                    root.currentModification = null;
                    refresh();
                }
            }));
        }
    });

    private UILabel waitLabel = new UILabel("Beat Len:", 16),
            drumVolumeLabel = new UILabel("Drum Vol:", 16);
    private UINumberBox waitBox = new UINumberBox(0, 16),
            drumVolumeBox = new UINumberBox(0, 16);

    // -- major

    private UISplitterLayout major;
    private UIDynamicProxy toolVS = new UIDynamicProxy();

    // Temp.Keybinder
    private int waitingKey = -1;
    private HashMap<Integer, Integer> bindings = new HashMap<Integer, Integer>();

    public NotesheetGame(final Game root) {
        super(640, 480);
        this.root = root;

        // -- UI Pre-Init --

        noteDisplay = new UINotesDisplay(root.coreFile, (aBoolean) -> {
            root.playerLock.lock();
            root.playerControls.setCurrentFrame(root.playerControls.getCurrentFrame() + (aBoolean ? -1 : 1));
            root.playerLock.unlock();
        });
        final UIElement playerControls = new UIPlayerControls(root.playerControls);
        UIElement trackAndPlayC = new UISplitterLayout(topPanC, playerControls, false, 0);

        // Something of a clean-up

        UIElement fileControlsL0I0 = UISplitterLayout.produceSideAlignedList(false, false, loadFile, saveFile);
        UISplitterLayout fileControlsL0I1 = new UISplitterLayout(new UISplitterLayout(increaseTime, decreaseTime, false, 0.5), timeFac, false, 0);
        UISplitterLayout fileControlsL0I2 = new UISplitterLayout(new UISplitterLayout(undo, redo, false, 0.5), clearDoBuffer, false, 0.5);
        UISplitterLayout fileControlsL0I3 = new UISplitterLayout(waitLabel, waitBox, false, 0);
        UISplitterLayout fileControlsL0I4 = new UISplitterLayout(drumVolumeLabel, drumVolumeBox, false, 0);
        UIElement fileControlsL0I5 = UISplitterLayout.produceSideAlignedList(false, false, configFS, configP, configX);

        UIElement fileControls = UISplitterLayout.produceSideAlignedList(
                false, false,
                fileControlsL0I0,
                new UIPublicPanel(16, 0),
                fileControlsL0I1,
                new UIPublicPanel(16, 0),
                fileControlsL0I2,
                new UIPublicPanel(16, 0),
                fileControlsL0I3,
                new UIPublicPanel(16, 0),
                fileControlsL0I4,
                new UIPublicPanel(16, 0),
                fileControlsL0I5,
                new UIPublicPanel(16, 0)
        );

        UIElement minor = new UISplitterLayout(toolVS, fileControls, true, 1);
        UIElement ts = new UISplitterLayout(trackScroll,  trackAndPlayC, true, 0);
        major = new UISplitterLayout(ts, minor, true, 0);

        // -- Wait/drumvol

        waitBox.onEdit = new Runnable() {
            @Override
            public void run() {
                root.startModifications("Beat Length");
                root.coreFile.musicWait = (int) waitBox.getNumber();
                root.endModifications();
            }
        };
        drumVolumeBox.onEdit = new Runnable() {
            @Override
            public void run() {
                root.startModifications("Drum Volume");
                root.coreFile.drumVolume = (int) drumVolumeBox.getNumber();
                root.endModifications();
                root.playerControls.regenWaveforms();
            }
        };

        // -- Track Header Editors

        for (int i = 0; i < 3; i++)
            trackHeaderEditor[i] = new UIWaveTrackHeaderEditor(root, root.coreFile.waveTracks[i]);

        // -- Refresh to initialize the above
        refresh();

        // Shush.
        UIPublicPanel dummy = new UIPublicPanel(640, 12 * (8 + 24)) {
            UILabel label;
            String textBuffer = "PiyoPiyoJ " + LibPiyo.VERSION + "\nBrought to you by:\n"
                    + "20kdc\n"
                    + "Jazz Jackalope, for the piano-keys text patch, and testing\n"
                    + "zxin, for more testing.\n"
                    + "Indirectly, Studio Pixel, for the sounds I put into piyoDrums.bin\n"
                    + "(Please don't sue me?)\n\n\n"
                    + "Hopefully I didn't miss anything, so with that -\n"
                    + " I now end the message.";

            @Override
            public void initialize() {
                label = new UILabel("", 16);
                layoutAddElement(label);
            }

            @Override
            public void layoutRunImpl() {
                Size s = getSize();
                label.setForcedBounds(this, new Rect(8, s.height - (label.getWantedSize().height + 8), s.width - 16, label.getWantedSize().height));
            }

            @Override
            public void update(double deltaTime, boolean selected, IPeripherals peripherals) {
                super.update(deltaTime, selected, peripherals);
                label.setText(textBuffer);
            }
        };
        dummy.imageScale = true;
        dummy.imageSW = 320;
        dummy.imageSH = 192;
        trackHeaderEditor[3] = dummy;

        // -- Tool Controls

        final NotesheetGame toolCallbackNG = this;

        noteDisplay.toolCallback = new UINotesDisplay.NoteClickCallback() {
            @Override
            public void onClick(int frameIndex, int pos, int button) {
                root.startModifications("Tool use");
                selectedTool.onClick(frameIndex, pos, button, toolCallbackNG);
            }

            @Override
            public void onDrag(int frameIndex, int pos) {
                selectedTool.onDrag(frameIndex, pos, toolCallbackNG);
            }

            @Override
            public void onRelease() {
                root.endModifications();
            }

            @Override
            public void onFingerClick(int i, int button) {
                root.playerControls.playNote(noteDisplay.track, i);
                if (button == 3)
                    waitingKey = i;
            }

            @Override
            public void prePC() {
                root.startModifications("Pan change");
            }

            @Override
            public void postPC() {
                root.endModifications();
            }
        };

        // -- Tools --

        selectTool(new NoteEditTool());

        layoutAddElement(noteDisplay);
        for (int i = 0; i < trackHeaderEditor.length; i++) {
            layoutAddElement(trackHeaderEditor[i]);
            layoutSetElementVis(trackHeaderEditor[i], false);
        }
        layoutAddElement(major);

        layoutRunImpl();

        // right click on fingers to prepare for binding, use ESC to get table for src.
        bindings.put(15, 2);
        bindings.put(16, 5);
        bindings.put(17, 8);
        bindings.put(18, 11);
        bindings.put(19, 14);
        bindings.put(20, 17);
        bindings.put(21, 20);
        bindings.put(22, 23);
        bindings.put(29, 1);
        bindings.put(30, 4);
        bindings.put(31, 7);
        bindings.put(32, 10);
        bindings.put(33, 13);
        bindings.put(34, 16);
        bindings.put(35, 19);
        bindings.put(36, 22);
        bindings.put(41, 0);
        bindings.put(42, 0);
        bindings.put(43, 3);
        bindings.put(44, 6);
        bindings.put(45, 9);
        bindings.put(46, 12);
        bindings.put(47, 15);
        bindings.put(48, 18);
        bindings.put(49, 21);
    }

    private void attachmentTrackHead(boolean b) {
        layoutSetElementVis(noteDisplay, !b);
        for (int i = 0; i < trackHeaderEditor.length; i++)
            layoutSetElementVis(trackHeaderEditor[i], b && (i == noteDisplay.track));
        if (editTrack.state == b)
            layoutRunImpl();
    }

    public void selectTool(EditTool editTool) {
        selectedTool = editTool;
        toolVS.dynProxySet(editTool.toolOpened(this));
    }

    @Override
    public void update(double deltaTime, boolean selected, IPeripherals peripherals) {
        // -- Update UI Elements --

        undo.setText(root.undoBuffer.size() > 0 ? "Undo" : "----");
        redo.setText(root.redoBuffer.size() > 0 ? "Redo" : "----");

        final int mfl = root.coreFile.getFrameCount();
        if (trackScroll.scrollPoint != lastScrollPoint) {
            root.playerLock.lock();
            root.playerControls.setCurrentFrame(Math.max((int) (trackScroll.scrollPoint * mfl), 0));
            root.playerLock.unlock();
        }
        int refCurrentFrame = (int) Math.floor(root.playerControls.getFrameInterpolatedPrecise());
        lastScrollPoint = trackScroll.scrollPoint = refCurrentFrame / ((double) mfl);

        keybindTickTime -= deltaTime;
        if (keybindTickTime <= 0) {
            keybindTickTime = 0.1d;
            if (peripherals instanceof IDesktopPeripherals) {
                IDesktopPeripherals idp = (IDesktopPeripherals) peripherals;
                if (idp.isKeyDown(IGrInDriver.VK_LEFT)) {
                    root.playerLock.lock();
                    root.playerControls.setCurrentFrame(refCurrentFrame - 1);
                    root.playerLock.unlock();
                }
                if (idp.isKeyDown(IGrInDriver.VK_RIGHT)) {
                    root.playerLock.lock();
                    root.playerControls.setCurrentFrame(refCurrentFrame + 1);
                    root.playerLock.unlock();
                }
            }
        }

        noteDisplay.xOffset = refCurrentFrame - 1;
        track1.state = noteDisplay.track == 0;
        track2.state = noteDisplay.track == 1;
        track3.state = noteDisplay.track == 2;
        trackP.state = noteDisplay.track == 3;


        for (int i = 0; i < noteDisplay.fingerInvert.length; i++) {
            noteDisplay.fingerInvert[i] = false;
            if (waitingKey == i)
                noteDisplay.fingerInvert[i] = (((long) GaBIEn.getTime()) & 1) == 0;
        }

        if (peripherals instanceof IDesktopPeripherals) {
            IDesktopPeripherals idp = (IDesktopPeripherals) peripherals;
            if (waitingKey != -1) {
                HashSet<Integer> hsi = idp.activeKeys();
                for (Integer i : hsi) {
                    bindings.put(i, waitingKey);
                    waitingKey = -1;
                    break;
                }
            }
            for (Map.Entry<Integer, Integer> kvp : bindings.entrySet()) {
                if (idp.isKeyJustPressed(kvp.getKey())) {
                    if (!selectedTool.keyboardNotifyShouldSuppress())
                        root.playerControls.playNote(noteDisplay.track, kvp.getValue());
                }
                if (idp.isKeyDown(kvp.getKey()))
                    noteDisplay.fingerInvert[kvp.getValue()] = true;
            }
            if (idp.isKeyJustPressed(IGrInDriver.VK_ESCAPE)) {
                System.err.println("// right click on fingers to prepare for binding, use ESC to get table for src.");
                LinkedList<Integer> k = new LinkedList<Integer>(bindings.keySet());
                Collections.sort(k);
                for (Integer i : k)
                    System.err.println("bindings.put(" + i + ", " + bindings.get(i) + ");");
            }
        }

        super.update(deltaTime, selected, peripherals);
    }

    private void refresh() {
        waitBox.setNumber(root.coreFile.musicWait);
        drumVolumeBox.setNumber(root.coreFile.drumVolume);
        ((UIWaveTrackHeaderEditor) trackHeaderEditor[0]).refresh();
        ((UIWaveTrackHeaderEditor) trackHeaderEditor[1]).refresh();
        ((UIWaveTrackHeaderEditor) trackHeaderEditor[2]).refresh();
    }

    @Override
    protected void layoutRunImpl() {
        int noteDisplayHeight = noteDisplay.getWantedSize().height;
        for (int i = 0; i < trackHeaderEditor.length; i++)
            noteDisplayHeight = Math.max(noteDisplayHeight, trackHeaderEditor[i].getWantedSize().height);
        Size s = getSize();
        Rect rb = new Rect(0, 0, s.width, noteDisplayHeight);
        if (editTrack.state) {
            trackHeaderEditor[noteDisplay.track].setForcedBounds(this, rb);
        } else {
            noteDisplay.setForcedBounds(this, rb);
        }
        major.setForcedBounds(this, new Rect(0, noteDisplayHeight, s.width, s.height - noteDisplayHeight));
    }
}
