/*
 * Part of the PiyoPiyoJ project.
 * (PiyoPiyo Java)
 *
 *        DO WHATEVER YOU WANT TO PUBLIC LICENSE
 *                    Version 2, December 2004
 *
 * Modified by 20kdc <asdd2808@gmail.com> to remove expletives
 * Original copyright (C) 2004 Sam Hocevar <sam@hocevar.net>
 *
 * Everyone is permitted to copy and distribute verbatim or modified
 * copies of this license document, and changing it is allowed as long
 * as the name is changed.
 *
 *            DO WHATEVER YOU WANT TO PUBLIC LICENSE
 *   TERMS AND CONDITIONS FOR COPYING, DISTRIBUTION AND MODIFICATION
 *
 *  0. You just DO WHATEVER YOU WANT TO.
 */

package piyopiyoj;

import java.util.concurrent.locks.ReentrantLock;

import libpiyo.PiyoPiyoFile;
import libpiyo.PiyoPiyoPlayer;
import libpiyo.SimpleMixer;

/**
 * Holder for player control information
 * Created on March 2nd, 2018
 */
public class PlayerControls {
    private boolean playing;
    public boolean reverse, halfSpeed;
    public final ReentrantLock playerLock;
    private final PiyoPiyoPlayer target;
    // reference track stuff is accessed in Game.java
    public boolean useRefTrack = true;
    public short[] refTrack = new short[1];
    public int refTrackPtr = 0;
    /**
     * Handle with care!
     */
    public final PiyoPiyoFile coreFile;
    private long withinBeatPositionLastUpdatedTime;

    public PlayerControls(ReentrantLock pl, PiyoPiyoFile file, SimpleMixer mixer, int baseSoundId) {
        playerLock = pl;
        target = new PiyoPiyoPlayer(file, mixer, baseSoundId);
        coreFile = file;
    }

    // NOTE: This runs on the audio thread.
    public void runTick(double deltaTime) {
        double speed = halfSpeed ? (deltaTime / 2.0d) : deltaTime;
        if (playing)
            target.updateBeats(reverse ? -speed : speed);
        withinBeatPositionLastUpdatedTime = System.currentTimeMillis();
    }

    public void setPlaying(boolean value) {
        playerLock.lock();
        playing = value;
        target.withinBeatPosition = 0.0d;
        withinBeatPositionLastUpdatedTime = System.currentTimeMillis();
        updateRefTrackPtr();
        playerLock.unlock();
    }

    private void updateRefTrackPtr() {
        // the -1 is to account for withinBeatPosition
        int direction = reverse ? -1 : 1;
        double exactSample = (target.currentFrame - direction) * coreFile.beatTime() * 22050d;
        int res = ((int) exactSample) * 2;
        refTrackPtr = res;
    }

    public boolean getPlaying() {
        return playing;
    }

    public void setCurrentFrame(int frame) {
        playerLock.lock();
        target.setCurrentFrame(frame);
        withinBeatPositionLastUpdatedTime = System.currentTimeMillis();
        updateRefTrackPtr();
        playerLock.unlock();
    }

    public int getCurrentFrame() {
        return target.currentFrame;
    }

    public void resetSynth() {
        playerLock.lock();
        target.resetSynth();
        playerLock.unlock();
    }

    public void regenWaveforms() {
        playerLock.lock();
        target.regenWaveforms();
        playerLock.unlock();
    }

    /**
     * Gets the precise interpolated beat. Yay!
     */
    public double getFrameInterpolatedPrecise() {
        playerLock.lock();
        double res;
        if (playing) {
            // penalty to avoid snapbacks
            // the interpolation is mainly meant to hide "buffer jank"
            double speed = (halfSpeed ? 0.5d : 1.0d) * 0.8d;
            speed = reverse ? -speed : speed;
            long movementMillis = System.currentTimeMillis() - withinBeatPositionLastUpdatedTime;
            double movementSeconds = (movementMillis / 1000d) * speed;
            double musicWaitTime = target.piyoFile.beatTime();
            res = target.currentFrame + ((target.withinBeatPosition + movementSeconds) / musicWaitTime);
        } else {
            res = target.currentFrame;
        }
        playerLock.unlock();
        return res;
    }

    public void playNote(int track, int i) {
        playerLock.lock();
        target.playNote(track, i);
        playerLock.unlock();
    }

    public boolean getHandleLoop() {
        return target.handleLoop;
    }

    public void setHandleLoop(boolean state) {
        target.handleLoop = state;
    }
}
