/*
 * Part of the PiyoPiyoJ project.
 * (PiyoPiyo Java)
 *
 *        DO WHATEVER YOU WANT TO PUBLIC LICENSE 
 *                    Version 2, December 2004 
 *
 * Modified by 20kdc <asdd2808@gmail.com> to remove expletives
 * Original copyright (C) 2004 Sam Hocevar <sam@hocevar.net> 
 *
 * Everyone is permitted to copy and distribute verbatim or modified 
 * copies of this license document, and changing it is allowed as long 
 * as the name is changed. 
 *
 *            DO WHATEVER YOU WANT TO PUBLIC LICENSE 
 *   TERMS AND CONDITIONS FOR COPYING, DISTRIBUTION AND MODIFICATION 
 *
 *  0. You just DO WHATEVER YOU WANT TO.
 */
package piyopiyoj;

import gabien.GaBIEn;
import gabien.audio.IRawAudioDriver;
import gabien.ui.*;
import gabien.ui.elements.*;
import gabien.ui.layouts.UISplitterLayout;
import libpiyo.PiyoPiyoFile;
import libpiyo.PiyoPiyoPlayer;
import libpiyo.PiyoPiyoPlayingSample;
import libpiyo.SimpleMixer;
import piyopiyoj.games.NotesheetGame;

import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.LinkedList;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;

import org.eclipse.jdt.annotation.NonNull;

/**
 * Used to be ripped straight out of IME,
 * but the 'Game' pattern really doesn't work that well when mixing in gabien.ui components.
 * If I port IME, I'll probably make a way to 'host' gabien.ui components without going all-in,
 * but PiyoPiyoJ is too UI-heavy for that to really be viable
 */
public class Game {
    public static final String FILE_X2 = "PiyoPiyoJ.X2";
    public static final String FILE_X3 = "PiyoPiyoJ.X3";
    public static final String FILE_FS = "PiyoPiyoJ.FS";
    
    private UIGame outerGame;
    public final SimpleMixer coreMixer = new SimpleMixer(PiyoPiyoPlayer.SOUND_ID_COUNT, 1);
    public final PiyoPiyoFile coreFile = new PiyoPiyoFile();
    public String rememberFileName = "Blank.pmd";
    public final ReentrantLock playerLock = new ReentrantLock();
    public final PlayerControls playerControls = new PlayerControls(playerLock, coreFile, coreMixer, 0);
    private final WindowCreatingUIElementConsumer worldCreator = new WindowCreatingUIElementConsumer();

    public LinkedList<PiyoPiyoFile> undoBuffer = new LinkedList<>();
    public LinkedList<PiyoPiyoFile> redoBuffer = new LinkedList<>();

    public String currentModification = null;

    private double recommendedDeltaTime() {
        if (!playerControls.getPlaying())
            return 0.05d;
        // This had a lock in it, but locking is causing issues.
        // This is read-only so it's not an issue.
        double beatTime = coreFile.beatTime();
        // this used to attempt to sync with audio thread, but that caused FUN issues
        if (playerControls.halfSpeed)
            beatTime *= 2;
        return beatTime / 8d;
    }

    public void runFrame() {
        double wakeUpTime = recommendedDeltaTime();
        while (wakeUpTime > 0.05d)
            wakeUpTime /= 2.0d;
        double deltaTime = GaBIEn.endFrame(wakeUpTime);

        if (worldCreator.runningWindows().size() == 0)
            GaBIEn.ensureQuit();

        try {
            worldCreator.runTick(deltaTime);
        } catch (Exception e) {
            if (currentModification != null) {
                currentModification = null;
                if (undoBuffer.size() > 0)
                    coreFile.copyFrom(undoBuffer.removeLast());
            }
            String saveState = "Data has been lost.";
            if (saveBackup())
                saveState = "Data saved to 'PPJ-BACKUP.pmd'.";
            try {
                switchError(e, new NotesheetGame(this));
            } catch (Exception e2) {
                switchError(e, new UILabel("Exception occurred during recovery. " + saveState, 16));
            }
        }
    }

    private boolean saveBackup() {
        try {
            OutputStream os = GaBIEn.getOutFile("PPJ-BACKUP.pmd");
            coreFile.saveFile(os);
            os.close();
            return true;
        } catch (Exception e3) {
        }
        return false;
    }
    public void resetInitLauncher() throws IOException {
        outerGame = new UIGame();
        doWCAccept();
        NotesheetGame ng = new NotesheetGame(this);
        try {
            PiyoPiyoPlayingSample.ensureAllSamplesLoaded();
            switcher(ng);
        } catch (IOException ioe) {
            switchError(ioe, ng);
        }
        IRawAudioDriver rad = GaBIEn.getRawAudio();
        rad.setRawAudioSource(new IRawAudioDriver.IRawAudioSource() {
            @Override
            public void pullData(@NonNull short[] interleaved, int ofs, int frames) {
                int blk = 880;
                while (frames > blk) {
                    pullData(interleaved, ofs, blk);
                    ofs += blk * 2;
                    frames -= blk;
                }
                playerLock.lock();
                playerControls.runTick(frames / 22050d);
                coreMixer.pullData(interleaved, ofs, frames);
                int ptr = playerControls.refTrackPtr;
                if (playerControls.getPlaying()) {
                    int direction = playerControls.reverse ? -1 : 1;
                    if (playerControls.useRefTrack) {
                        short[] refTrack = playerControls.refTrack;
                        int ofs2 = ofs;
                        for (int i = 0; i < frames * 2; i++) {
                            int v = interleaved[ofs2];
                            if (ptr >= 0 && ptr < refTrack.length) {
                                v += refTrack[ptr] >> 1;
                                ptr += direction;
                            }
                            v >>= 1;
                            interleaved[ofs2++] = (short) v;
                        }
                    }
                    playerControls.refTrackPtr += frames * 2 * direction;
                }
                playerLock.unlock();
            }
        });
    }

    private void tsMakeFile(String str) {
        try {
            OutputStream os = GaBIEn.getOutFile(str);
            os.close();
        } catch (Exception ex) {
            // oops
        }
    }
    public void toggleScale(boolean positive) {
        if (positive) {
            if (GaBIEn.fileOrDirExists(FILE_X3)) {
                // nothing!
            } else if (GaBIEn.fileOrDirExists(FILE_X2)) {
                tsMakeFile(FILE_X3);
                GaBIEn.rmFile(FILE_X2);
            } else {
                tsMakeFile(FILE_X2);
            }
            saveBackup();
        } else {
            if (GaBIEn.fileOrDirExists(FILE_X3)) {
                GaBIEn.rmFile(FILE_X3);
                tsMakeFile(FILE_X2);
            } else if (GaBIEn.fileOrDirExists(FILE_X2)) {
                GaBIEn.rmFile(FILE_X2);
            } else {
                // nothing!
            }
        }
        worldCreator.forceRemove(outerGame);
        doWCAccept();
    }
    public void toggleFullscreen() {
        if (GaBIEn.fileOrDirExists(FILE_FS)) {
            GaBIEn.rmFile(FILE_FS);
        } else {
            tsMakeFile(FILE_FS);
        }
        worldCreator.forceRemove(outerGame);
        doWCAccept();
    }
    private void doWCAccept() {
        int scale = 1;
        if (GaBIEn.fileOrDirExists(FILE_X3)) {
            scale = 3;
        } else if (GaBIEn.fileOrDirExists(FILE_X2)) {
            scale = 2;
        }
        worldCreator.accept(outerGame, scale, GaBIEn.fileOrDirExists(FILE_FS));
    }

    public void switcher(UIElement returnGame) {
        outerGame.changeElement(returnGame);
    }
    public void fileDialog(boolean saving, final Consumer<String> recipient, String verb) {
        String here = GaBIEn.absolutePathOf("./");
        final UIElement currentGame = outerGame.getElement();
        outerGame.changeElement(new UILabel("Selecting a file: " + verb, 16));
        GaBIEn.startFileBrowser(here, saving, "", new Consumer<String>() {
            @Override
            public void accept(String t) {
                switcher(currentGame);
                recipient.accept(t);
            }
        });
    }

    public void switchError(Exception e, final UIElement notesheetGame) {
        String ef = "Exception occurred! Details follow...\n";
        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw);
        e.printStackTrace(pw);
        pw.flush();
        ef += sw.toString().replace("\r\n", "\n").replace("\t", "    ");
        switcher(new UISplitterLayout(new UILabel(ef, 16), new UITextButton("Continue", 16, new Runnable() {
            @Override
            public void run() {
                switcher(notesheetGame);
            }
        }), true, 1));
    }

    public void startModifications(String s) {
        redoBuffer.clear();
        undoBuffer.add(new PiyoPiyoFile(coreFile));
    }

    public void endModifications() {
        if (currentModification != null)
            currentModification = null;
    }

    public void undoModifications() {
        if (undoBuffer.size() > 0) {
            redoBuffer.add(new PiyoPiyoFile(coreFile));
            coreFile.copyFrom(undoBuffer.removeLast());
        }
    }

    public void redoModifications() {
        if (redoBuffer.size() > 0) {
            undoBuffer.add(new PiyoPiyoFile(coreFile));
            coreFile.copyFrom(redoBuffer.removeLast());
        }
    }
}
