/*
 * gabien-app-ppj - Editor/Player for 'PiyoPiyo' music files
 * Written starting in 2015 by contributors (see CREDITS.txt)
 * To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
 * You should have received a copy of the CC0 Public Domain Dedication along with this software. If not, see <http://creativecommons.org/publicdomain/zero/1.0/>.
 */
package libpiyo.export;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.LinkedList;

import gabien.GaBIEn;
import libpiyo.LibPiyo;
import libpiyo.PiyoPiyoFile;
import libpiyo.PiyoPiyoFrame;
import libpiyo.PiyoPiyoPlayingSample;
import libpiyo.PiyoPiyoWaveTrack;

public class XMOut {
    // Drum volume ratios
    public static final int DRUMVOL_RATIO_MUL_Q = 3;
    public static final int DRUMVOL_RATIO_MUL_L = 4;
    public static final int DRUMVOL_RATIO_DIV = 4;

    public static void main(String[] args) throws Exception {
        if (args.length != 3)
            throw new RuntimeException("Expects three arguments, source .pmd, target .xm, channel count");
        GaBIEn.initializeEmbedded();
        try {
            PiyoPiyoPlayingSample.ensureAllSamplesLoaded();
        } catch (Exception ex) {
            ex.printStackTrace();
        }
        PiyoPiyoFile ppf = new PiyoPiyoFile();
        ppf.loadFile(GaBIEn.getInFile(args[0]));
        int chc = Integer.valueOf(args[2]);
        OutputStream os = GaBIEn.getOutFile(args[1]);
        perform(os, ppf, GaBIEn.nameOf(args[0]), chc);
        os.close();
    }

    public static void perform(OutputStream os, PiyoPiyoFile ppf, String basename, int channels) throws IOException {
        // needs to be 64 because supposedly Pattern Break doesn't like you if you go over 64.
        // that's important because Pattern Break is the basis of the accurate loop technique used here.
        int rowsPerPattern = 64;
        // It's vitally important for looping that we don't emit any more patterns after the pattern with the "loop row".
        // The idea is to use the Restart Position combined with Pattern Break.
        int patterns = (ppf.loopEnd + rowsPerPattern - 1) / rowsPerPattern;
        int instruments = 9;
        // Attempt to determine best tempo/BPM.
        int tempo = 4;
        int bpm = 125;
        double target = ppf.beatTime();
        double closeness = Double.MAX_VALUE;
        for (int xTempo = 1; xTempo <= 31; xTempo++) {
            for (int xBPM = 32; xBPM <= 255; xBPM++) {
                double tc = Math.abs(target - secondsPerRowXM(xTempo, xBPM));
                if (tc < closeness) {
                    tempo = xTempo;
                    bpm = xBPM;
                    closeness = tc;
                }
            }
        }
        // Write the header.
        putHeader(os, channels, patterns, instruments, tempo, bpm, ppf.loopStart / rowsPerPattern, basename);
        // Normalize instrument volumes (yes, this can muck with quiet PMDs).
        // This needs to be done early because drums use volume overrides.
        int[] iVolumes = new int[] {
                ppf.waveTracks[0].volume,
                ppf.waveTracks[1].volume,
                ppf.waveTracks[2].volume,
                ppf.drumVolume
        };
        int maxVolume = 0;
        for (int i = 0; i < iVolumes.length; i++)
            if (maxVolume < iVolumes[i])
                maxVolume = iVolumes[i];
        for (int i = 0; i < iVolumes.length; i++)
            iVolumes[i] = ((iVolumes[i] * 64) / maxVolume);
        // Start writing patterns.
        int channelRotation = 0;
        int currentFrame = 0;
        // Keep track of panning - note panning is whatever the panning value was for that track at the time the note was played.
        int[] panning = new int[] {3, 3, 3, 3};
        int[] panningTable = new int[] {
                128 - 96,
                128 - 64,
                128 - 32,
                128,
                128 + 32,
                128 + 64,
                128 + 96
        };
        for (int i = 0; i < patterns; i++) {
            // Setup a blank pattern to work with.
            XMCell[] cells = new XMCell[rowsPerPattern * channels];
            for (int j = 0; j < cells.length; j++)
                cells[j] = new XMCell();
            // Handle all the frames in this pattern.
            int rowCellBase = 0;
            for (int j = 0; j < rowsPerPattern; j++) {
                for (int k = 0; k < 4; k++) {
                    PiyoPiyoFrame cf = ppf.getFrameOrNull(currentFrame, k);
                    if (cf == null)
                        continue;
                    // panning
                    if (cf.panValue != 0)
                        panning[k] = cf.panValue - 1;
                    for (int n = 0; n < 24; n++) {
                        if (!cf.hitNotes[n])
                            continue;
                        // Finally, something to do!
                        XMCell cell = cells[rowCellBase + channelRotation];
                        if (k != 3) {
                            cell.clear();
                            int noteAdj = n + (ppf.waveTracks[k].octave * 12) + 1;
                            if (noteAdj < 1)
                                noteAdj = 1;
                            if (noteAdj > 127)
                                noteAdj = 127;
                            cell.note = (byte) noteAdj;
                            cell.instrument = (byte) (k + 1);
                        } else {
                            int drumId = PiyoPiyoPlayingSample.drumTable[n / 2];
                            if (drumId == -1)
                                continue;
                            boolean louder = (n % 2) == 0;
                            cell.clear();
                            cell.note = 49;
                            cell.instrument = (byte) (drumId + 4);
                            cell.volume = (byte) ((iVolumes[3] * (louder ? DRUMVOL_RATIO_MUL_L : DRUMVOL_RATIO_MUL_Q)) / DRUMVOL_RATIO_DIV);
                        }
                        cell.effect = 8;
                        cell.effectParam = (byte) panningTable[panning[k]];
                        channelRotation++;
                        channelRotation %= channels;
                    }
                }
                if (currentFrame == (ppf.loopEnd - 1)) {
                    // uses restart position + pattern break to hit the right spot every time
                    // note we don't clear the cell, but this will take priority over any existing effect (Pan) if we REALLY have to
                    XMCell ef1 = cells[rowCellBase + channelRotation];
                    ef1.effect = 0x0D;
                    int jumpRow = ppf.loopStart % rowsPerPattern;
                    int jumpRowDec = (jumpRow % 10) + ((jumpRow / 10) * 0x10);
                    ef1.effectParam = (byte) (jumpRowDec);
                    channelRotation++;
                    channelRotation %= channels;
                }
                currentFrame++;
                rowCellBase += channels;
            }
            // Now write it out.
            XMCell.putPattern(os, cells, channels);
        }
        // Write instruments.
        for (int i = 0; i < 3; i++) {
            PiyoPiyoWaveTrack wt = ppf.waveTracks[i];
            LinkedList<XMInstrumentOut.Point> points = XMInstrumentConversion.convIPoints(wt, bpm);
            XMInstrumentConversion.rmOverlappingPoints(points);
            XMInstrumentConversion.complexStripIPoints(points);
            XMInstrumentOut.putInstrument(os, "Melody " + i, points, wt.waveForm, iVolumes[i]);
        }
        for (int i = 0; i < 6; i++) {
            short[] perc = PiyoPiyoPlayingSample.samples[i];
            if (perc == null)
                perc = new short[1];
            XMInstrumentOut.putInstrumentPercussion(os, "Drum " + i, perc);
        }
    }
    public static double secondsPerRowXM(int tempo, int bpm) {
        double rpm = (24d * bpm) / tempo;
        return 60d / rpm;
    }
    public static void putHeader(OutputStream os, int channels, int patterns, int instruments, int tempo, int bpm, int restartPosition, String basename) throws IOException {
        ByteBuffer bb = ByteBuffer.allocate(60 + 20 + 256);
        bb.order(ByteOrder.LITTLE_ENDIAN);
        // - 60 -
        bb.put("Extended Module: ".getBytes("UTF-8"));
        putTextInBB(bb, basename, 20);
        bb.put((byte) 0x1A);
        putTextInBB(bb, "PiyoPiyoJ " + LibPiyo.VERSION + " Export", 20);
        bb.putShort((short) 0x0104);
        // - 20 -
        bb.putInt(276); // header size
        bb.putShort((short) patterns); // song length
        bb.putShort((short) restartPosition); // restart position
        bb.putShort((short) channels); // channels
        bb.putShort((short) patterns); // patterns
        bb.putShort((short) instruments); // instruments
        bb.putShort((short) 1); // linear
        bb.putShort((short) tempo); // tempo
        bb.putShort((short) bpm); // BPM
        // - 256 -
        for (int i = 0; i < 256; i++)
            bb.put((byte) i);
        os.write(bb.array());
    }
    public static void putTextInBB(ByteBuffer bb, String text, int len) throws IOException {
        // MilkyTracker and Schism Tracker share a semi-custom codepage we'll call "codepage Impulse".
        // It's a derivative of CP437, and in practice using CP437 would usually be best for compatibility.
        // OpenMPT uses the system codepage.
        // *Keep in mind PiyoPiyo filenames are mostly in Japanese.*
        // So the correct choice is SHIFT-JIS, because that's the only situation where any of this actually works.
        String charset = "SHIFT_JIS";
        byte[] data = text.getBytes(charset);
        while (data.length > len) {
            text = text.substring(0, text.length() - 1);
            data = text.getBytes(charset);
        }
        bb.put(data);
        for (int i = data.length; i < len; i++)
            bb.put((byte) 32);
    }
}
