/*
 * gabien-app-ppj - Editor/Player for 'PiyoPiyo' music files
 * Written starting in 2015 by contributors (see CREDITS.txt)
 * To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
 * You should have received a copy of the CC0 Public Domain Dedication along with this software. If not, see <http://creativecommons.org/publicdomain/zero/1.0/>.
 */
package libpiyo.export;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.LinkedList;

public class XMInstrumentOut {
    // Instrument basis
    public static final int INSTRUMENT_HEADER_SIZE = 29;
    public static final int INSTRUMENT_HEADER2_SIZE = 214;
    public static final int INSTRUMENT_SH_SIZE = 40;

    public static void putInstrumentHeader(ByteBuffer bb, String name, int samples, int totalSize) throws IOException {
        // 29 bytes - instrument header
        bb.putInt(totalSize); // isize
        XMOut.putTextInBB(bb, name, 22);
        bb.put((byte) 0);
        bb.putShort((short) samples);
    }
    public static void putInstrument(OutputStream os, String name, LinkedList<Point> points, int[] waveForm, int vol) throws IOException {
        ByteBuffer bb = ByteBuffer.allocate(INSTRUMENT_HEADER_SIZE + INSTRUMENT_HEADER2_SIZE + INSTRUMENT_SH_SIZE + waveForm.length);
        bb.order(ByteOrder.LITTLE_ENDIAN);
        // 29 bytes - instrument header
        putInstrumentHeader(bb, name, 1, INSTRUMENT_HEADER_SIZE + INSTRUMENT_HEADER2_SIZE);
        // 223 bytes - instrument header, part 2
        bb.putInt(INSTRUMENT_SH_SIZE); // SH size
        for (int i = 0; i < 96; i++)
            bb.put((byte) 0); // note mapping table
        // -- Volume points --
        int pointCount = points.size();
        if (pointCount > 12)
            throw new RuntimeException("maximum of 12 points allowed");
        for (Point p : points) {
            // Used points
            bb.putShort((short) p.x);
            bb.putShort((short) p.y);
        }
        for (int i = pointCount; i < 12; i++) {
            // Empty points
            bb.putShort((short) 0);
            bb.putShort((short) 0);
        }
        // -- Other points --
        for (int i = 0; i < 24; i++)
            bb.putShort((short) 0); // points
        // -- The rest --
        bb.put((byte) pointCount); // vpc
        bb.put((byte) 2); // ppc
        bb.put((byte) 0); // vsp
        bb.put((byte) 0); // vlsp
        bb.put((byte) 0); // vlep
        bb.put((byte) 0); // psp
        bb.put((byte) 0); // plsp
        bb.put((byte) 0); // plep
        bb.put((byte) 1); // vt - enable volume envelope
        bb.put((byte) 0); // pt
        bb.put((byte) 0); // vit
        bb.put((byte) 0); // vis
        bb.put((byte) 0); // vid
        bb.put((byte) 0); // vir
        bb.putShort((short) 0); // vfadeout
        bb.putShort((short) 0); // Res.
        // 40 bytes - sample header
        bb.putInt(waveForm.length);
        bb.putInt(0);
        bb.putInt(waveForm.length);
        bb.put((byte) vol); // Volume
        bb.put((byte) 0); // Finetune
        bb.put((byte) 0x01); // Flags
        bb.put((byte) 128); // Panning
        bb.put((byte) 48); // RNN
        bb.put((byte) 0); // Res.
        XMOut.putTextInBB(bb, "Melody", 22);
        // bytes of sample data
        int running = 0;
        for (int i = 0; i < waveForm.length; i++) {
            int val = waveForm[i];
            bb.put((byte) (val - running));
            running = val;
        }
        // Done!
        os.write(bb.array());
    }
    public static void putInstrumentPercussion(OutputStream os, String name, short[] data) throws IOException {
        ByteBuffer bb = ByteBuffer.allocate(INSTRUMENT_HEADER_SIZE + INSTRUMENT_HEADER2_SIZE + INSTRUMENT_SH_SIZE + (data.length * 2));
        bb.order(ByteOrder.LITTLE_ENDIAN);
        // 29 bytes - instrument header
        putInstrumentHeader(bb, name, 1, INSTRUMENT_HEADER_SIZE + INSTRUMENT_HEADER2_SIZE);
        if (bb.position() != INSTRUMENT_HEADER_SIZE)
            throw new RuntimeException("Bad instrument H1 alloc");
        // 223 bytes - instrument header, part 2
        bb.putInt(INSTRUMENT_SH_SIZE); // SH size
        for (int i = 0; i < 96; i++)
            bb.put((byte) 0); // note mapping table
        for (int i = 0; i < 48; i++)
            bb.putShort((short) 0); // points
        bb.put((byte) 2); // vpc
        bb.put((byte) 2); // ppc
        bb.put((byte) 0); // vsp
        bb.put((byte) 0); // vlsp
        bb.put((byte) 0); // vlep
        bb.put((byte) 0); // psp
        bb.put((byte) 0); // plsp
        bb.put((byte) 0); // plep
        bb.put((byte) 0); // vt
        bb.put((byte) 0); // pt
        bb.put((byte) 0); // vit
        bb.put((byte) 0); // vis
        bb.put((byte) 0); // vid
        bb.put((byte) 0); // vir
        bb.putShort((short) 0); // vfadeout
        bb.putShort((short) 0); // Res.
        if (bb.position() != (INSTRUMENT_HEADER_SIZE + INSTRUMENT_HEADER2_SIZE))
            throw new RuntimeException("Bad instrument H2 alloc (pos=" + bb.position() + ")");
        // 40 bytes - sample header
        bb.putInt(data.length * 2);
        bb.putInt(0);
        bb.putInt(0);
        bb.put((byte) 64); // Volume
        bb.put((byte) 0); // Finetune
        bb.put((byte) 0x10); // Flags
        bb.put((byte) 128); // Panning
        bb.put((byte) 17); // RNN
        bb.put((byte) 0); // Res.
        XMOut.putTextInBB(bb, "Percussion", 22);
        if (bb.position() != (INSTRUMENT_HEADER_SIZE + INSTRUMENT_HEADER2_SIZE + INSTRUMENT_SH_SIZE))
            throw new RuntimeException("Bad instrument SH alloc");
        // N*2 bytes - sample data
        short running = 0;
        for (int i = 0; i < data.length; i++) {
            short smp = data[i];
            bb.putShort((short) (smp - running));
            running = smp;
        }
        if (bb.position() != bb.capacity())
            throw new RuntimeException("Bad instrument percussion alloc " + bb.position());
        // Finally done
        os.write(bb.array());
    }
    public static class Point {
        public short x, y;
        public Point(short tx, short ty) {
            x = tx;
            y = ty;
        }
    }
}
