/*
 * gabien-app-ppj - Editor/Player for 'PiyoPiyo' music files
 * Written starting in 2015 by contributors (see CREDITS.txt)
 * To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
 * You should have received a copy of the CC0 Public Domain Dedication along with this software. If not, see <http://creativecommons.org/publicdomain/zero/1.0/>.
 */
package libpiyo.export;

import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedList;

import libpiyo.PiyoPiyoWaveTrack;
import libpiyo.export.XMInstrumentOut.Point;

public class XMInstrumentConversion {
    // Envelope length control
    public static final long ENVLEN_RATIO_MUL = 10;
    public static final long ENVLEN_RATIO_DIV = 8000;
    // Divisor applied since the input to this is the 0-63 envelope position.
    // Applied in the hope of some theory which makes these maths sensible.
    public static final long ENVLEN_SRC_DIV = 64;
    // Divisor applied to convert "beats per minute" to "beats per second".
    public static final long ENVLEN_BPM2BPS_DIV = 60;
    // All the constant divisors multiplied together.
    // ENVLEN_RATIO obviously corresponds to it's other half, BPM2BPS corresponds to bpm, and ENVLEN_SRC_DIV corresponds to envPos.
    public static final long ENVLEN_ALL_DIV = ENVLEN_RATIO_DIV * ENVLEN_BPM2BPS_DIV * ENVLEN_SRC_DIV;

    public static LinkedList<Point> convIPoints(PiyoPiyoWaveTrack wt, int bpm) {
        LinkedList<Point> points = new LinkedList<Point>();
        // note that divisors for these parts are described in ENVLEN_ALL_DIV
        long baseMultiplier = ENVLEN_RATIO_MUL * bpm * wt.length;
        short lastX = 0;
        for (int envPos = 0; envPos < wt.envelope.length; envPos++) {
            // envelope points
            // note: BPM controls speed of envelopes, too, so higher BPM = longer envs required
            long epm = envPos * baseMultiplier;
            short x = (short) (epm / ENVLEN_ALL_DIV);
            short y = (short) (wt.envelope[envPos] >> 1);
            points.add(new Point(x, y));
            lastX = x;
        }
        // collapse to 0
        points.add(new Point((short) (lastX + 1), (short) 0));
        return points;
    }

    public static void naiveStripIPoints(LinkedList<Point> points) {
        if (points.size() <= 12)
            return;
        LinkedList<Point> intermediate = new LinkedList<Point>();
        // Idea is that im1 is the last value before the cut,
        //  which we want to keep so sharp end cuts work.
        // Might not work out in practice but should at least help termination.
        int im1 = points.size() - 2;
        int jump = im1 / 10;
        for (int i = 0; i <= im1; i += jump) {
            if (intermediate.size() >= 11)
                break;
            intermediate.add(points.get(i));
        }
        intermediate.add(points.getLast());
        points.clear();
        points.addAll(intermediate);
    }

    public static void complexStripIPoints(LinkedList<Point> points) {
        // obviously no need to do this
        if (points.size() <= 12)
            return;
        // Alright, so, here's the deal.
        // First and foremost, get a profile from before we start removing points.
        short[] originalProfile = emulatePoints(points);
        // Secondly, start removing points.
        while (true) {
            int pointCount = points.size();
            if (pointCount <= 12)
                break;
            int bestRemovalIndex = 1;
            long bestRemovalError = Long.MAX_VALUE;
            // The first point must never be removed on pain of XM spec violation.
            // The last point shouldn't be removed because it silences the voice.
            for (int i = 1; i < pointCount - 1; i++) {
                LinkedList<Point> attempt = new LinkedList<Point>(points);
                attempt.remove(i);
                long error = errorMetric(originalProfile, emulatePoints(attempt));
                if (error < bestRemovalError) {
                    bestRemovalIndex = i;
                    bestRemovalError = error;
                }
            }
            points.remove(bestRemovalIndex);
        }
    }

    // Error metric for emulatePoints
    // a is original
    public static long errorMetric(short[] a, short[] b) {
        long total = 0;
        for (int i = 0; i < a.length; i++) {
            long va = a[i];
            long vb = b[i];
            long vc = va - vb;
            if (vc < 0)
                vc = -vc;
            total += vc;
        }
        return total;
    }

    // Emulation of point data conversion
    public static short[] emulatePoints(LinkedList<Point> points) {
        short[] sim = new short[65536];
        if (points.size() == 0)
            return sim;
        Point pointA = points.get(0);
        int i = 0;
        if (points.size() != 1) {
            int pointBIndex = 1;
            Point pointB = points.get(1);
            for (; i < sim.length; i++) {
                if (pointB.x == i) {
                    // advance
                    pointA = pointB;
                    pointBIndex++;
                    if (pointBIndex == points.size())
                        break;
                    pointB = points.get(pointBIndex);
                }
                // clamped lerp time
                int relative = i - pointA.x;
                int relativeLen = pointB.x - (int) pointA.x;
                if (relative < 0)
                    relative = 0;
                // shouldn't happen, just paranoia
                if (relative > relativeLen)
                    relative = relativeLen;
                int relativeInv = relativeLen - relative;
                sim[i] = (short) (((pointA.y * relativeInv) + (pointB.y * relative)) / relativeLen);
            }
        }
        for (; i < sim.length; i++)
            sim[i] = pointA.y;
        return sim;
    }

    public static void rmOverlappingPoints(LinkedList<Point> points) {
        Collections.sort(points, new Comparator<Point>() {
            public int compare(Point o1, Point o2) {
                if (o1.x < o2.x)
                    return -1;
                if (o1.x > o2.x)
                    return 1;
                return 0;
            }
        });
        short lastX = -1;
        for (int i = 0; i < points.size(); i++) {
            short thisX = points.get(i).x;
            if (thisX > lastX) {
                lastX = thisX;
                continue;
            }
            points.remove(i);
            i--;
        }
    }
}
