/*
 * gabien-app-ppj - Editor/Player for 'PiyoPiyo' music files
 * Written starting in 2015 by contributors (see CREDITS.txt)
 * To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
 * You should have received a copy of the CC0 Public Domain Dedication along with this software. If not, see <http://creativecommons.org/publicdomain/zero/1.0/>.
 */
package libpiyo.export;

import gabien.media.audio.*;
import gabien.media.audio.AudioIOSource.SourceS16;
import gabien.media.audio.fileio.WavIO;
import libpiyo.PiyoPiyoFile;
import libpiyo.PiyoPiyoPlayer;
import libpiyo.SimpleMixer;

import java.io.IOException;
import java.io.OutputStream;

public class WavOut {
    public static void perform(OutputStream fos, PiyoPiyoFile coreFile, int startFrame, int ignoreFrame, int frameCount) throws IOException {
        final SimpleMixer sm = new SimpleMixer(PiyoPiyoPlayer.SOUND_ID_COUNT, 2);
        final PiyoPiyoPlayer iPlayer = new PiyoPiyoPlayer(coreFile, sm, 0);
        final int sampleRate = sm.sampleRate;
        final double time = 1d / sampleRate;
        // Alter pitch. An explanation of this:
        // PiyoPiyoPlayer's timescale is just in seconds (you advance it and it updates the mixer channels).
        // However, the pitch values it calculates assumes a 22050hz "base frequency".
        // So the pitch has to be "virtually lowered" to account for this.
        iPlayer.pitchScale = 22050f / sampleRate;
        iPlayer.setCurrentFrame(startFrame);
        // Regarding frameCount: Some complex loops could fool a more flexible approach.
        int ignoreSampleCount = (int) (ignoreFrame * sampleRate * iPlayer.piyoFile.beatTime());
        final int sampleFrameCount = (int) (frameCount * sampleRate * iPlayer.piyoFile.beatTime());

        int maxSkipTimestep = 32;
        short[] tmpInterleaved = new short[maxSkipTimestep * 2];
        for (int i = 0; i < ignoreSampleCount; i += maxSkipTimestep) {
            iPlayer.updateBeats(time * maxSkipTimestep);
            // Stereo data.
            sm.pullData(tmpInterleaved, 0, Math.min(ignoreSampleCount - i, maxSkipTimestep));
        }

        WavIO.writeWAV(fos, new SourceS16(new AudioIOCRSet(2, sampleRate)) {
            @Override
            public int frameCount() {
                return sampleFrameCount;
            }

            @Override
            public void nextFrames(short[] buffer, int at, int frames) {
                while (frames > 0) {
                    iPlayer.updateBeats(time);
                    sm.pullData(buffer, at, 1);
                    at += 2;
                    frames--;
                }
            }
        }, AudioIOFormat.F_S16);
    }
}
