/*
 * gabien-app-ppj - Editor/Player for 'PiyoPiyo' music files
 * Written starting in 2015 by contributors (see CREDITS.txt)
 * To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
 * You should have received a copy of the CC0 Public Domain Dedication along with this software. If not, see <http://creativecommons.org/publicdomain/zero/1.0/>.
 */
package libpiyo.export;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedList;

import libpiyo.PiyoPiyoFile;
import libpiyo.PiyoPiyoFrame;

public class MidOut {
    public static void perform(OutputStream fos, PiyoPiyoFile ppf) throws IOException {
        DataOutputStream dos = new DataOutputStream(fos);
        fos.write('M');
        fos.write('T');
        fos.write('h');
        fos.write('d');
        dos.writeInt(6);
        dos.writeShort(0);
        dos.writeShort(1);

        // ROOT VALUES:
        int dtPB = 500;
        // microseconds per beat...
        int uspb = 0x07A120;

        if (Parameter.MSCOMPAT.value == 1) {
            uspb = ppf.musicWait * 1000;
            dtPB = ppf.musicWait;
        }

        // SUB VALUES:
        double dtPMS = dtPB / (uspb / 1000d);
        int deltaTimeMusicWait = (int) (dtPMS * ppf.musicWait);
        System.err.println("DTMW " + deltaTimeMusicWait + " at " + uspb + "uspb/" + dtPB + "dtpb/" + dtPMS + " calced dtPMS");


        // Calculate DeltaTimes per "beat"
        // At 500mspb, let's get 1ms resolution
        dos.writeShort(dtPB);

        // --

        fos.write('M');
        fos.write('T');
        fos.write('r');
        fos.write('k');

        ByteArrayOutputStream track = new ByteArrayOutputStream();

        // the track

        writeVLI(track, 0);
        track.write(0xFF);
        track.write(0x58);
        track.write(4);
        track.write(4);
        track.write(2);
        track.write(24);
        track.write(8);

        writeVLI(track, 0);
        track.write(0xFF);
        track.write(0x51);
        track.write(3);
        track.write(uspb >> 16);
        track.write(uspb >> 8);
        track.write(uspb);

        // We use 3 primary channels
        // 0, 1, 2

        int[] channels = {
                Parameter.A_C1.value,
                Parameter.A_C2.value,
                Parameter.A_C3.value,
                9
        };
        int[] programs = {
                Parameter.P_C1.value,
                Parameter.P_C2.value,
                Parameter.P_C3.value
        };
        int[] volumes = {
                Parameter.V_C1.value,
                Parameter.V_C2.value,
                Parameter.V_C3.value,
                Parameter.V_CP.value
        };

        for (int i = 0; i < 4; i++) {
            if (i != 3) {
                writeVLI(track, 0);
                track.write(0xC0 + i); // program
                track.write(programs[i]);
            }

            writeVLI(track, 0);
            track.write(0xB0 + channels[i]); // control
            track.write(0x07); // Volume (0-127)
            track.write(volumes[i]);
        }

        int[] balanceTranslator = {
                0,
                0, // L
                21,
                42,
                64, // Mid
                86,
                107,
                127, // R
        };

        LinkedList<MidiEvent> midiEvents = new LinkedList<MidiEvent>();

        MidiEvent[] triggerNoteOff = new MidiEvent[24 * 4];

        // Ok, so there's some crazy stuff that goes on here

        int startF = 0;
        int endF = ppf.getFrameCount();

        boolean cutting = false; // strict-time
        boolean continuous = false; // requires cutting

        if (Parameter.MODE.value == 0) {
            startF = 0;
            endF = ppf.loopEnd; // leads into...
            cutting = true;
        } else if (Parameter.MODE.value == 1) {
            startF = ppf.loopStart; // this, which leads into itself
            endF = ppf.loopEnd;
            continuous = true;
            cutting = true;
        } else if (Parameter.MODE.value == 2) {
            // this is the default!
        }

        for (int i = startF; i < endF; i++) {
            int currTimeFrame = (i - startF) * deltaTimeMusicWait;
            for (int channelNo = 0; channelNo < 4; channelNo++) {
                PiyoPiyoFrame frame = ppf.getFrameOrNull(i, channelNo);
                if (frame == null)
                    continue;

                // firstly, change pan
                if (frame.panValue != 0)
                    midiEvents.add(MidiEvent.pan(currTimeFrame, channels[channelNo], balanceTranslator[frame.panValue]));

                for (int j = 0; j < frame.hitNotes.length; j++) {
                    int tnoIndex = j + (channelNo * 24);
                    int mappedNote = -1;
                    int envelopeDT = deltaTimeMusicWait / 2;
                    if (channelNo < 3) {
                        int ets = 500;
                        int no = 24;
                        if (channelNo == 0) {
                            ets = Parameter.E_C1.value;
                            no = Parameter.D_C1.value;
                        } else if (channelNo == 1) {
                            ets = Parameter.E_C2.value;
                            no = Parameter.D_C2.value;
                        } else if (channelNo == 2) {
                            ets = Parameter.E_C3.value;
                            no = Parameter.D_C3.value;
                        }
                        mappedNote = no + (ppf.waveTracks[channelNo].octave * Parameter.EOW.value) + j;
                        envelopeDT = (int) ((ppf.waveTracks[channelNo].length / 22050d) * dtPMS * ets);
                        if (envelopeDT < 1)
                            envelopeDT = 1;
                    } else {
                        int[] mapping = {
                                Parameter._DS0.value,
                                Parameter._DS0.value,
                                Parameter._DS2.value,
                                Parameter._DS2.value,
                                Parameter._DS4.value,
                                Parameter._DS4.value,
                                -1,
                                -1,
                                Parameter._DS6.value, // maybe 42
                                Parameter._DS6.value,
                                Parameter._DS8.value,
                                Parameter._DS8.value,
                                Parameter._DSA.value,
                                Parameter._DSA.value,
                                -1,
                                -1,
                                -1,
                                -1,
                                -1,
                                -1,
                                -1,
                                -1,
                                -1,
                                -1
                        };
                        mappedNote = mapping[j];
                    }
                    if (mappedNote != -1) {
                        if (frame.hitNotes[j]) {
                            if (triggerNoteOff[tnoIndex] != null)
                                triggerNoteOff[tnoIndex].exactTime = Math.min(triggerNoteOff[tnoIndex].exactTime, currTimeFrame);
                            int velocity = 0x40;
                            if (channelNo == 3) {
                                if ((j & 1) == 0) {
                                    velocity = Parameter.V_PA.value;
                                } else {
                                    velocity = Parameter.V_PB.value;
                                }
                            }
                            midiEvents.add(MidiEvent.note(currTimeFrame, channels[channelNo], mappedNote, true, velocity));
                            midiEvents.add(triggerNoteOff[tnoIndex] = MidiEvent.note(currTimeFrame + envelopeDT, channels[channelNo], mappedNote, false, 0x40));
                        }
                    }
                }
            }
        }
        // System.err.println("LTF " + lastTimeFrame);

        int endTime = (endF - startF) * deltaTimeMusicWait;

        midiEvents.add(MidiEvent.end(endTime));

        // aplaymidi -d 0 -p 129:0 TidepoolI.mid Tidepool.mid Tidepool.mid Tidepool.mid Tidepool.mid
        if (cutting) {
            if (continuous) {
                for (MidiEvent me : midiEvents) {
                    if (me.subOrder != 3) {
                        if (me.subOrder == 0)
                            me.subOrder = -1;
                        me.exactTime %= endTime;
                    }
                }
            } else {
                for (MidiEvent me : new LinkedList<MidiEvent>(midiEvents))
                    if (me.exactTime > endTime)
                        midiEvents.remove(me);
            }
        }


        // Prepare to move note off events that are in group -1 under various circumstances
        int[] usedNoteMap = new int[16 * 128];
        for (int i = 0; i < usedNoteMap.length; i++)
            usedNoteMap[i] = -1;

        for (int i = 0; i < 2; i++) {
            Collections.sort(midiEvents, new Comparator<MidiEvent>() {
                @Override
                public int compare(MidiEvent midiEvent, MidiEvent t1) {
                    if (midiEvent.exactTime < t1.exactTime)
                        return -1;
                    if (midiEvent.exactTime > t1.exactTime)
                        return 1;
                    if (midiEvent.subOrder < t1.subOrder)
                        return -1;
                    if (midiEvent.subOrder > t1.subOrder)
                        return 1;
                    return 0;
                }
            });
            for (MidiEvent me : midiEvents) {
                if ((me.subOrder == 0) || (me.subOrder == 2)) {
                    int unn = usedNoteMap[me.getCNI()];
                    if (unn == -1)
                        usedNoteMap[me.getCNI()] = me.exactTime;
                }
                if (me.subOrder == -1) {
                    int unn = usedNoteMap[me.getCNI()];
                    if (unn != -1)
                        me.exactTime = Math.min(unn, me.exactTime);
                }
            }
        }

        int writerTime = 0;
        for (MidiEvent me : midiEvents) {
            writeVLI(track, me.exactTime - writerTime);
            writerTime = me.exactTime;
            track.write(me.data);
        }

        // done with track

        dos.writeInt(track.size());
        track.writeTo(fos);

        // end of file
    }

    private static void writeVLI(OutputStream fos, int i) throws IOException {
        if (i >= (1 << 21))
            fos.write(0x80 | ((i >> 21) & 0x7F));
        if (i >= (1 << 14))
            fos.write(0x80 | ((i >> 14) & 0x7F));
        if (i >= (1 << 7))
            fos.write(0x80 | ((i >> 7) & 0x7F));
        fos.write(i & 0x7F);
    }

    public static void setup(String text) {
        // detect some of Pixel's songs and set at least half-decent programs
        if (text.toLowerCase().endsWith("tidepool.pmd")) {
            Parameter.P_C1.value = 0;
            Parameter.P_C2.value = 7;
            Parameter.P_C3.value = 38;
        } else if (text.toLowerCase().endsWith("ikachan.pmd")) {
            Parameter.P_C1.value = 38;
            Parameter.P_C2.value = 0;
            Parameter.P_C3.value = 38;
        }
    }

    public enum Parameter {
        // Channel P relies on the Channel 9 system
        A_C1(0, 0, 15, "Channel 1 MIDIChannel", true),
        P_C1(0, 0, 127, "Channel 1 Program", true),
        E_C1(1, 500, 2000, "Channel 1 EnvTimeScale", true),
        V_C1(0, 127, 127, "Channel 1 Volume", true),
        D_C1(0, 24, 127, "Channel 1 Note Offset", true),

        A_C2(0, 1, 15, "Channel 2 MIDIChannel", true),
        P_C2(0, 0, 127, "Channel 2 Program", true),
        E_C2(1, 1000, 2000, "Channel 2 EnvTimeScale", true),
        V_C2(0, 127, 127, "Channel 2 Volume", true),
        D_C2(0, 24, 127, "Channel 2 Note Offset", true),

        A_C3(0, 2, 15, "Channel 3 MIDIChannel", true),
        P_C3(0, 0, 127, "Channel 3 Program", true),
        E_C3(1, 500, 2000, "Channel 3 EnvTimeScale", true),
        V_C3(0, 127, 127, "Channel 3 Volume", true),
        D_C3(0, 24, 127, "Channel 3 Note Offset", true),

        V_CP(0, 127, 127, "Channel P Volume", false),
        V_PA(0, 64, 127, "Channel P! Velocity", false),
        V_PB(0, 32, 127, "Channel P. Velocity", false),

        _DS0(0, 35, 127, "X1 Note", false),
        _DS2(0, 36, 127, "X2 Note", false),
        _DS4(0, 39, 127, "X3 Note", false),
        _DS6(0, 54, 127, "HH Note", false),
        _DS8(0, 33, 127, "Cl Note", false),
        _DSA(0, 44, 127, "Cs Note", false),

        EOW(0, 12, 127, "Effective Octave Size", false),
        MODE(0, 2, 2, "Cut Mode\n0: Intro\n1: Loop\n2: All", false),
        MSCOMPAT(0, 0, 1, "MuseScore Compat.\n1 to enable", false);

        public final int min, max;
        public final String text;
        public final boolean left;
        public int value;

        Parameter(int i, int i1, int i2, String s, boolean l) {
            left = l;
            min = i;
            value = i1;
            max = i2;
            text = s;
        }
    }

    public static class MidiEvent {
        public int exactTime;
        public byte[] data;
        // -1: BACKDATED-OFF
        // 0: OFF 1: PAN/OTHER 2: ON 3: END
        public int subOrder = 0;
        public static MidiEvent note(int time, int channel, int note, boolean on, int velocity) {
            MidiEvent me = new MidiEvent();
            me.exactTime = time;
            me.subOrder = on ? 2 : 0;
            me.data = new byte[] {
                    (byte) (0x80 + channel + (on ? 0x10 : 0)),
                    (byte) note,
                    (byte) velocity
            };
            return me;
        }
        public static MidiEvent pan(int time, int channel, int pan) {
            MidiEvent me = new MidiEvent();
            me.exactTime = time;
            me.subOrder = 1;
            me.data = new byte[] {
                    (byte) (0xB0 + channel),
                    (byte) 0x08,
                    (byte) pan
            };
            return me;
        }
        public static MidiEvent end(int time) {
            MidiEvent me = new MidiEvent();
            me.exactTime = time;
            me.subOrder = 3;
            me.data = new byte[] {
                    (byte) 0xFF,
                    (byte) 0x2F,
                    (byte) 0x00
            };
            return me;
        }

        public int getCNI() {
            return (data[0] & 0xF) | ((data[1]) << 4);
        }
    }
}
