/*
 * gabien-app-ppj - Editor/Player for 'PiyoPiyo' music files
 * Written starting in 2015 by contributors (see CREDITS.txt)
 * To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
 * You should have received a copy of the CC0 Public Domain Dedication along with this software. If not, see <http://creativecommons.org/publicdomain/zero/1.0/>.
 */

package libpiyo;

import org.eclipse.jdt.annotation.NonNull;

import gabien.uslx.append.MathsX;

/**
 * This iteration of the mixer attempts to roughly emulate Sound.cpp as used in PiyoPiyo(/Player). 
 * Frequency support is missing, which should boost performance.
 * Created on 09/12/15. Reworked 15th November, 2025.
 */
public class SimpleMixer {
    public static final int CANONICAL_SAMPLE_RATE = 22050;
    // Converts volume to 0-256 range.
    private static final int[] VOLCONV = new int[301];
    static {
        for (int i = 0; i <= 300; i++) {
            // Note that 'volume 0' is nowhere near 0.
            float f = (i / 600.0f) + 0.5f;
            // This is intended to approximate the decibel curve that's (presumably) really being used here.
            f *= f;
            f *= 256.0f;
            // TODO this should probably be a curve
            VOLCONV[i] = (int) f;
        }
    }

    public final boolean[] active;
    public final short[][] sound;
    public final int[] pos;

    /**
     * These volume values are in the Sound.cpp units. (0-300, safe to clamp)
     * To summarize how this works: It seems DirectSound doesn't like to amplify sounds, only duck them.
     * See dsound.h DSBVOLUME_MIN / DSBVOLUME_MAX.
     */
    public final int[] volume;

    /**
     * These pan values are based on, but not the same as, the Studio Pixel 0-256-512 units.
     * -256 is left, 256 is right.
     */
    public final int[] pan;

    public int totalVolumeState;

    /**
     * Actual expected sample rate
     */
    public final int sampleRate;
    /**
     * Multiplier on 22050hz (the 'canonical' sample rate)
     */
    public final int sampleRateMul;
    public final int decayFracMul;

    public SimpleMixer(int channelCount, int sampleRateMul) {
        this.sampleRate = CANONICAL_SAMPLE_RATE * sampleRateMul;
        this.sampleRateMul = sampleRateMul;
        this.decayFracMul = sampleRateMul * 0x100;
        active = new boolean[channelCount];
        sound = new short[channelCount][];
        pos = new int[channelCount];
        volume = new int[channelCount];
        pan = new int[channelCount];
    }

    public void pullData(@NonNull short[] interleaved, int ofs, int frames) {
        int lim = ofs + (frames * 2);
        for (int i = ofs; i < lim; i += 2) {
            // 1/256 frac.
            int totalL = 0;
            int totalR = 0;
            int totalVolume = 0;
            for (int j = 0; j < active.length; j++) {
                if (!active[j])
                    continue;

                int cPos = pos[j];
                short[] cSnd = sound[j];
                if (cPos >= cSnd.length) {
                    active[j] = false;
                    continue;
                }
                short sample = cSnd[cPos];
                pos[j] = cPos + 1;

                int cVol = VOLCONV[MathsX.clamp(volume[j], 0, 300)];
                totalVolume += cVol;
                // So there's a bit of complexity in how this works, but I think it should hold.
                // cPan : -256 - 256
                int cPan = MathsX.clamp(pan[j], -256, 256);
                // Problem: PiyoPiyo doesn't drive this all the way yet the pan is very extreme.
                // Solution for now is a multiplier.
                cPan = (cPan * 3) / 2;
                // The basic energy-preserving principle here can be confirmed with panTest.pmd
                int cPanInv = -cPan;
                int panL = cPan <= 0 ? 256 : (256 - cPan);
                int panR = cPanInv <= 0 ? 256 : (256 - cPanInv);
                int pannedL = (sample * panL) >> 8;
                int pannedR = (sample * panR) >> 8;
                totalL += pannedL * cVol;
                totalR += pannedR * cVol;
            }
            // Don't perform ducking if the total volume is below or equal to the unit.
            if (totalVolume < 0x100)
                totalVolume = 0x100;
            // decay logic
            totalVolumeState = Math.max(totalVolumeState - 1, totalVolume * decayFracMul);
            int totalVolumeRelaxed = totalVolumeState / decayFracMul;
            // Tweak
            totalL *= 3;
            totalR *= 3;
            // Ducking / bring values back into the -32768-32767 range.
            totalL /= totalVolumeRelaxed;
            totalR /= totalVolumeRelaxed;
            interleaved[i] = (short) MathsX.clamp(totalL, -32768, 32767);
            interleaved[i + 1] = (short) MathsX.clamp(totalR, -32768, 32767);
        }
    }
}
