/*
 * Part of the GaBIEn project.
 * (Graphics and Basic Input Engine)
 *
 *        DO WHATEVER YOU WANT TO PUBLIC LICENSE 
 *                    Version 2, December 2004 
 *
 * Modified by 20kdc <asdd2808@gmail.com> to remove expletives
 * Original copyright (C) 2004 Sam Hocevar <sam@hocevar.net> 
 *
 * Everyone is permitted to copy and distribute verbatim or modified 
 * copies of this license document, and changing it is allowed as long 
 * as the name is changed. 
 *
 *            DO WHATEVER YOU WANT TO PUBLIC LICENSE 
 *   TERMS AND CONDITIONS FOR COPYING, DISTRIBUTION AND MODIFICATION 
 *
 *  0. You just DO WHATEVER YOU WANT TO.
 */
package libpiyo;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

public class PiyoPiyoWaveTrack {
    // These should be stored as bytes, but Java bytes are signed, and I don't want to get near that insanity...
    // Further information: Using Java bytes for waveForm *does* make sense. For envelope, not so much.
    public int[] waveForm = new int[256];
    public int[] envelope = new int[64];
    public int volume, length, octave, icon;

    public PiyoPiyoWaveTrack(int tt) {
        reset(tt);
    }

    public void copyFrom(PiyoPiyoWaveTrack waveTrack) {
        for (int i = 0; i < waveForm.length; i++)
            waveForm[i] = waveTrack.waveForm[i];
        for (int i = 0; i < envelope.length; i++)
            envelope[i] = waveTrack.envelope[i];
        octave = waveTrack.octave;
        icon = waveTrack.icon;
        length = waveTrack.length;
        volume = waveTrack.volume;
    }

    public void reset(int trackType) {
        if (trackType == 0) {
            for (int i = 0; i < waveForm.length / 2; i++) {
                waveForm[i] = 96;
                waveForm[i + 128] = 192;
            }
        } else if (trackType == 1) {
            for (int i = 0; i < waveForm.length / 2; i++) {
                waveForm[i] = i;
                waveForm[i + 128] = 255 - i;
            }
        } else {
            for (int i = 0; i < waveForm.length; i++) {
                int sineMod = (int) (Math.sin((i / (double) waveForm.length) * 2 * Math.PI) * 99.9999);
                waveForm[i] = (128 + sineMod) ^ 0x80;
            }
        }
        // As for this, just make something up
        for (int i = 0; i < envelope.length; i++)
            envelope[i] = 0;
        for (int i = 0; i < (envelope.length / 2); i++)
            envelope[i] = 127;
        // Fancy maths for fancy envelopes
        int[] oldEnv = new int[envelope.length];
        for (int i = 0; i < 64; i++) {
            System.arraycopy(envelope, 0, oldEnv, 0, envelope.length);
            for (int j = 1; j < envelope.length - 1; j++) {
                envelope[j] = (oldEnv[j - 1] + oldEnv[j] + oldEnv[j + 1]) / 3;
                if (j > 16) {
                    envelope[j] += oldEnv[j - 16] / 2;
                    envelope[j] /= 2;
                }
            }
        }
        octave = 1;
        icon = 0;
        length = 11000;
        volume = 250;
    }

    public void read(InputStream is) throws IOException {
        octave = PiyoPiyoFile.read8(is);
        icon = PiyoPiyoFile.read8(is);
        PiyoPiyoFile.read8(is);
        PiyoPiyoFile.read8(is);
        length = PiyoPiyoFile.read32(is);
        volume = PiyoPiyoFile.read32(is);
        PiyoPiyoFile.read32(is);
        PiyoPiyoFile.read32(is);
        PiyoPiyoFile.readArray(is, waveForm);
        PiyoPiyoFile.readArray(is, envelope);
    }

    public void write(OutputStream os) throws IOException {
        os.write(octave);
        os.write(icon);
        os.write(0);
        os.write(0);
        PiyoPiyoFile.write32(os, length);
        PiyoPiyoFile.write32(os, volume);
        PiyoPiyoFile.write32(os, 0);
        PiyoPiyoFile.write32(os, 0);
        PiyoPiyoFile.writeArray(os, waveForm);
        PiyoPiyoFile.writeArray(os, envelope);
    }

    /**
     * This is the note generator, aka the sketchiest part of the project.
     * To avoid directly copying code, we still reimplement everything somewhat manually (and so in 16-bit!) but still. Grr!
     */
    public short[] generateSample(int note, int sampleRateMul) {
        int sampleRate = SimpleMixer.CANONICAL_SAMPLE_RATE * sampleRateMul;
        int octaveAdj = octave & 31;
        note += octaveAdj * 12;
        short[] data = new short[length * sampleRateMul];
        int advanceFP = (int) (((8363f * Math.pow(2.0f, note / 12.0f)) / sampleRate) * 256);
        int waveformPosFP = 0;
        for (int i = 0; i < data.length; i++) {
            int envPos = (i * 64) / data.length;
            int wfpInt = waveformPosFP >> 8;
            int val = ((waveForm[wfpInt % waveForm.length] & 0xFF) ^ 0x80) - 0x80;
            val *= envelope[envPos] * 2;
            data[i] = (short) val;
            waveformPosFP += advanceFP;
        }
        return data;
    }
}
