/*
 * Part of the GaBIEn project.
 * (Graphics and Basic Input Engine)
 *
 *        DO WHATEVER YOU WANT TO PUBLIC LICENSE 
 *                    Version 2, December 2004 
 *
 * Modified by 20kdc <asdd2808@gmail.com> to remove expletives
 * Original copyright (C) 2004 Sam Hocevar <sam@hocevar.net> 
 *
 * Everyone is permitted to copy and distribute verbatim or modified 
 * copies of this license document, and changing it is allowed as long 
 * as the name is changed. 
 *
 *            DO WHATEVER YOU WANT TO PUBLIC LICENSE 
 *   TERMS AND CONDITIONS FOR COPYING, DISTRIBUTION AND MODIFICATION 
 *
 *  0. You just DO WHATEVER YOU WANT TO.
 */
package libpiyo;

import java.io.IOException;
import java.io.InputStream;

import org.eclipse.jdt.annotation.Nullable;

public class PiyoPiyoPlayer {
    public static final int SOUND_ID_COUNT = 24 * 4;

    public double withinBeatPosition = 0;
    public final PiyoPiyoFile piyoFile;
    public SimpleMixer sound;
    public int baseSoundId;
    public int currentFrame = 0;
    // If false, then loop start/end is ignored.
    public boolean handleLoop = true;

    public float pitchScale = 1;

    public PiyoPiyoPlayer(InputStream fis, SimpleMixer sound, int baseSoundId) throws IOException {
        this.sound = sound;
        this.baseSoundId = baseSoundId;
        piyoFile = new PiyoPiyoFile();
        piyoFile.loadFile(fis);
        resetSynth();
    }

    public PiyoPiyoPlayer(PiyoPiyoFile file, SimpleMixer sound, int baseSoundId) {
        this.sound = sound;
        this.baseSoundId = baseSoundId;
        piyoFile = file;
        resetSynth();
    }

    public void resetSynth() {
        for (int i = 0; i < 96; i++) {
            sound.active[baseSoundId + i] = false;
            sound.pan[baseSoundId + i] = 0;
        }
        regenWaveforms();
    }

    /**
     * For live editing.
     * Use with care; you may want resetSynth instead.
     */
    public void regenWaveforms() {
        int soundId = baseSoundId;
        for (int trackId = 0; trackId < 4; trackId++) {
            for (int note = 0; note < 24; note++) {
                if (trackId == 3) {
                    // Drum tracks
                    sound.sound[soundId] = PiyoPiyoPlayingSample.getSample(note, sound.sampleRateMul);
                    if ((note & 1) == 0) {
                        sound.volume[soundId] = piyoFile.drumVolume;
                    } else {
                        sound.volume[soundId] = (piyoFile.drumVolume * 70) / 100;
                    }
                } else {
                    PiyoPiyoWaveTrack wt = piyoFile.waveTracks[trackId];
                    sound.sound[soundId] = wt.generateSample(note, sound.sampleRateMul);
                    sound.volume[soundId] = wt.volume;
                }
                soundId++;
            }
        }
    }

    // Update beats. Can be run in reverse.
    public void updateBeats(double deltaTime) {
        withinBeatPosition += deltaTime;
        double musicWaitTime = piyoFile.beatTime();
        while (withinBeatPosition < 0) {
            withinBeatPosition += musicWaitTime;
            for (int i = 0; i < 4; i++)
                handleBeat(piyoFile.getFrameOrNull(currentFrame, i), i);
            currentFrame--;
            if (currentFrame < 0)
                setCurrentFrame(piyoFile.loopEnd);
        }
        int pfc = piyoFile.getFrameCount();
        while (withinBeatPosition > musicWaitTime) {
            withinBeatPosition -= musicWaitTime;
            for (int i = 0; i < 4; i++)
                handleBeat(piyoFile.getFrameOrNull(currentFrame, i), i);
            currentFrame++;
            if ((currentFrame >= pfc) || (handleLoop && (currentFrame >= piyoFile.loopEnd)))
                setCurrentFrame(piyoFile.loopStart);
        }
    }

    public void setCurrentFrame(int value) {
        currentFrame = value;
        if (currentFrame < 0)
            currentFrame = 0;
        int pfc = piyoFile.getFrameCount();
        if (currentFrame >= pfc)
            currentFrame = pfc - 1;
        withinBeatPosition = 0;
    }

    private void handleBeat(@Nullable PiyoPiyoFrame piyoPiyoFrame, int track) {
        if (piyoPiyoFrame == null)
            return;
        if (piyoPiyoFrame.panValue != 0)
            panTrack(track, piyoPiyoFrame.panValue);
        for (int i = 0; i < 24; i++)
            if (piyoPiyoFrame.hitNotes[i])
                playNote(track, i);
    }

    public void playNote(int track, int i) {
        int soundId = baseSoundId + (track * 24) + i;
        sound.active[soundId] = true;
        sound.pos[soundId] = 0;
    }

    public void panTrack(int track, int ppPanValue) {
        int panValue = 0;
        if (ppPanValue == 1)
            panValue = -160;
        if (ppPanValue == 2)
            panValue = -76;
        if (ppPanValue == 3)
            panValue = -32;
        // 4 is centre
        if (ppPanValue == 5)
            panValue = 32;
        if (ppPanValue == 6)
            panValue = 76;
        if (ppPanValue == 7)
            panValue = 164;
        // System.out.println("Pantest: " + ppPanValue + " -> " + panValue);
        int soundId = baseSoundId + (track * 24);
        for (int i = 0; i < 24; i++)
            sound.pan[soundId++] = panValue;
    }
}
