/*
 * Part of the GaBIEn project.
 * (Graphics and Basic Input Engine)
 *
 *        DO WHATEVER YOU WANT TO PUBLIC LICENSE 
 *                    Version 2, December 2004 
 *
 * Modified by 20kdc <asdd2808@gmail.com> to remove expletives
 * Original copyright (C) 2004 Sam Hocevar <sam@hocevar.net> 
 *
 * Everyone is permitted to copy and distribute verbatim or modified 
 * copies of this license document, and changing it is allowed as long 
 * as the name is changed. 
 *
 *            DO WHATEVER YOU WANT TO PUBLIC LICENSE 
 *   TERMS AND CONDITIONS FOR COPYING, DISTRIBUTION AND MODIFICATION 
 *
 *  0. You just DO WHATEVER YOU WANT TO.
 */
package libpiyo;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import org.eclipse.jdt.annotation.Nullable;

/**
 * This is now reasonably thread-safe as of 18th June, 2023.
 */
public final class PiyoPiyoFile {
    public final PiyoPiyoWaveTrack[] waveTracks = new PiyoPiyoWaveTrack[3];
    public int musicWait, drumVolume, loopStart, loopEnd;
    private PiyoPiyoFrame[] musicFrames;

    public PiyoPiyoFile() {
        waveTracks[0] = new PiyoPiyoWaveTrack(0);
        waveTracks[1] = new PiyoPiyoWaveTrack(1);
        waveTracks[2] = new PiyoPiyoWaveTrack(2);
        reset();
    }

    public PiyoPiyoFile(PiyoPiyoFile other) {
        this();
        copyFrom(other);
    }

    public synchronized void copyFrom(PiyoPiyoFile other) {
        waveTracks[0].copyFrom(other.waveTracks[0]);
        waveTracks[1].copyFrom(other.waveTracks[1]);
        waveTracks[2].copyFrom(other.waveTracks[2]);
        musicWait = other.musicWait;
        drumVolume = other.drumVolume;
        loopStart = other.loopStart;
        loopEnd = other.loopEnd;
        musicFrames = new PiyoPiyoFrame[other.musicFrames.length];
        for (int i = 0; i < musicFrames.length; i++)
            musicFrames[i] = new PiyoPiyoFrame(other.musicFrames[i]);
    }

    // For optimization/expansion tasks.
    // (This is used in the editor so you never run out of space)
    public synchronized void resize(int targetLength) {
        // System.err.println("PPF: Resizing from " + (musicFrames.length / 4) + " to " + targetLength);
        PiyoPiyoFrame[] oldMusicFrames = musicFrames;
        PiyoPiyoFrame[] newMusicFrames = new PiyoPiyoFrame[targetLength * 4];
        int omfSection = (oldMusicFrames.length / 4);
        int nmfSection = targetLength;
        for (int i = 0; i < targetLength; i++) {
            if (i < omfSection) {
                newMusicFrames[i] = oldMusicFrames[i];
                newMusicFrames[i + nmfSection] = oldMusicFrames[i + omfSection];
                newMusicFrames[i + (nmfSection * 2)] = oldMusicFrames[i + (omfSection * 2)];
                newMusicFrames[i + (nmfSection * 3)] = oldMusicFrames[i + (omfSection * 3)];
            } else {
                newMusicFrames[i] = new PiyoPiyoFrame();
                newMusicFrames[i + nmfSection] = new PiyoPiyoFrame();
                newMusicFrames[i + (nmfSection * 2)] = new PiyoPiyoFrame();
                newMusicFrames[i + (nmfSection * 3)] = new PiyoPiyoFrame();
            }
        }
        musicFrames = newMusicFrames;
    }

    public int getFrameCount() {
        return musicFrames.length / 4;
    }

    public @Nullable PiyoPiyoFrame getFrameOrNull(int at, int channel) {
        int fd4 = musicFrames.length / 4;
        PiyoPiyoFrame[] cmf = musicFrames;
        if (at < 0)
            return null;
        if (at >= fd4)
            return null;
        return cmf[at + (fd4 * channel)];
    }

    // This is here so if this changes in future stuff doesn't break.
    // For now this value has been determined by random experimentation.
    public double beatTime() {
        // If the result is too long, increase, if the result is too short, decrease

        // This is *SUPPOSED* to be the beat time. For better or worse.
        // (Jazz Jackalope noted that 905.0d was actually making it run too slowly)
        return Math.max(musicWait / 1000.0d, 0.001d);
    }

    public synchronized void reset() {
        musicWait = 130;
        loopStart = 0;
        loopEnd = 0;
        int musicLength = 128;
        musicFrames = new PiyoPiyoFrame[musicLength * 4];
        for (int i = 0; i < waveTracks.length; i++)
            waveTracks[i].reset(i);
        drumVolume = 300;
        for (int i = 0; i < musicFrames.length; i++)
            musicFrames[i] = new PiyoPiyoFrame();
    }

    public synchronized void loadFile(InputStream is) throws IOException {
        if ((read8(is) != ((int) 'P')) || (read8(is) != ((int) 'M')) || (read8(is) != ((int) 'D')))
            throw new IOException("Not a PMD file!");
        read8(is);// song flags
        int pos = read32(is) - 0x418;
        musicWait = read32(is);
        loopStart = read32(is);
        loopEnd = read32(is);
        musicFrames = new PiyoPiyoFrame[read32(is) * 4];
        waveTracks[0].read(is);
        waveTracks[1].read(is);
        waveTracks[2].read(is);
        drumVolume = read32(is);
        is.skip(pos);
        for (int i = 0; i < musicFrames.length; i++) {
            musicFrames[i] = new PiyoPiyoFrame();
            musicFrames[i].read(is);
        }
    }

    public void saveFile(OutputStream os) throws IOException {
        os.write((int) 'P');
        os.write((int) 'M');
        os.write((int) 'D');
        os.write(0x80);
        write32(os, 0x418);
        write32(os, musicWait);
        write32(os, loopStart);
        write32(os, loopEnd);
        write32(os, musicFrames.length / 4);
        waveTracks[0].write(os);
        waveTracks[1].write(os);
        waveTracks[2].write(os);
        write32(os, drumVolume);
        for (int i = 0; i < musicFrames.length; i++)
            musicFrames[i].write(os);
    }

    protected static int read8(InputStream is) throws IOException {
        int i = is.read();
        if (i == -1)
            throw new IOException("Unexpected EOF");
        return i;
    }

    protected static int read32(InputStream is) throws IOException {
        int i = read8(is);
        i |= read8(is) << 8;
        i |= read8(is) << 16;
        i |= read8(is) << 24;
        return i;
    }

    protected static void readArray(InputStream is, int[] b) throws IOException {
        for (int i = 0; i < b.length; i++)
            b[i] = read8(is);
    }

    protected static void write32(OutputStream is, int i) throws IOException {
        is.write(i & 0xFF);
        is.write((i >> 8) & 0xFF);
        is.write((i >> 16) & 0xFF);
        is.write((i >> 24) & 0xFF);
    }

    protected static void writeArray(OutputStream is, int[] b) throws IOException {
        for (int i = 0; i < b.length; i++)
            is.write(b[i]);
    }

    public synchronized void trimEnd() {
        int origLen = musicFrames.length / 4;
        int bestLength = origLen;
        while (bestLength > 1) {
            if (loopStart == bestLength)
                break;
            if (loopEnd == bestLength)
                break;
            boolean empty = true;
            int lastFrameBase = bestLength - 1;
            for (int i = 0; i < 4; i++)
                if (!musicFrames[lastFrameBase + (origLen * i)].isEmpty())
                    empty = false;
            if (!empty)
                break;
            bestLength--;
        }
        resize(bestLength);
    }

    public PiyoPiyoFrame[] getAllFrames() {
        return musicFrames;
    }

    public synchronized void setAllFrames(PiyoPiyoFrame[] frames) {
        musicFrames = frames;
    }

    public synchronized PiyoPiyoFrame ensureFrame(int i, int track) {
        int fc = getFrameCount();
        if (i < 0)
            return new PiyoPiyoFrame();
        if (i >= fc)
            resize(i + 1);
        return getFrameOrNull(i, track);
    }
}
